<?php

namespace App\Controllers;
use \App\Libraries\Complementos;
use \App\Libraries\Scripts;
use \App\Libraries\Error;
use \App\Libraries\Alertas;
use \App\Libraries\Pdf;

require 'vendor/autoload.php';
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\IOFactory;
use PhpOffice\PhpSpreadsheet\Cell\Coordinate;

class Administracion extends BaseController
{

	public function __construct(){
		$this->usuarioModelo = new \App\Models\UsuarioModelo();
		$this->perfilModelo = new \App\Models\PerfilModelo();
		$this->subAccionModelo = new \App\Models\SubAccionModelo();
        $this->direccionModelo = new \App\Models\DireccionModel();
        $this->ubicacionAcceso = new \App\Models\DireccionUbicacionAccesoModel();
        $this->departamentoINEI = new \App\Models\INEIDepartamentoModel();
        $this->provinciaINEI = new \App\Models\INEIProvinciaModel();
        $this->distritoINEI = new \App\Models\INEIDistritoModel();
        $this->modulo = new \App\Models\ModuloModel();
        $this->moduloAcciones = new \App\Models\ModulosAccionesModel();
        $this->moduloRutas = new \App\Models\ModulosRutasModel();
        $this->moduloCabecera = new \App\Models\ModulosCabeceraModel();
        $this->perfilPermisos = new \App\Models\PerfilPermisosModelo();
        $this->perfilRutas = new \App\Models\PerfilRutasModelo();
        
        $this->sistemas = new \App\Models\SistemasModel();
		$this->departamentos = new \App\Models\DepartamentoModel();
        $this->areas = new \App\Models\AreaModel();
        $this->cargos = new \App\Models\CargosModel();
        
        
        $this->estructuraDirecciones = new \App\Models\EstructuraDireccionModel();
        $this->estructuraAreas = new \App\Models\EstructuraAreaModel();
        $this->estructuraSubareas = new \App\Models\EstructuraSubareaModel();
        $this->estructuraSubareaDelegaciones = new \App\Models\EstructuraSubareaDelegacionesModel();
        $this->estructuraDelegaciones = new \App\Models\EstructuraDelegacionModel();
	}

	public function obtenerINEIDepartamentos(){
    	/*
        * ----------------------
        * OBTIENE Y VALIDA DATOS
        * ----------------------
        */
        $listadoDepartamentos = $this->departamentoINEI->findAll();
        
        $departamentos=array();

		if (!empty($listadoDepartamentos)) {
            foreach ($listadoDepartamentos as $items) {
                $departamentos[] = array(
					'id' => (string) $items['id_dep_inei'],
         	        'departamento' => (string) $items['desc_dep_inei'],
                );
            }

            $resultado = array(
				'listadoDepartamentos' => json_encode($departamentos)
			);
        } else {
			$resultado = array(
				'descripcion' => 0,
			);
		}

        /*
         * ----------------------------
         * DATOS DE AJAX PRE PROCESADOS
         * ----------------------------
         */
        echo json_encode($resultado, JSON_NUMERIC_CHECK);		
	}

	public function obtenerINEIProvincias(){
    	/*
        * ----------------------
        * OBTIENE Y VALIDA DATOS
        * ----------------------
        */
        
		$idDepartamento = $this->request->getPost('idDepartamento');

        if($idDepartamento == 'TOTAL')
        {
            $listadoProvincias = $this->provinciaINEI->findAll();
        }
        else
        {
            $whereProvincia = array('id_dep_inei' => $idDepartamento);
            $listadoProvincias = $this->provinciaINEI->where($whereProvincia)->findAll();
        }
        
        $provincias=array();

		if (!empty($listadoProvincias)) {
            foreach ($listadoProvincias as $items) {
                $provincias[] = array(
					'id' => (string) $items['id_prov_inei'],
         	        'provincia' => (string) $items['desc_prov_inei'],
                );
            }

            $resultado = array(
				'listadoProvincias' => json_encode($provincias)
			);
        } else {
			$resultado = array(
				'descripcion' => 0,
			);
		}

        /*
         * ----------------------------
         * DATOS DE AJAX PRE PROCESADOS
         * ----------------------------
         */
        echo json_encode($resultado, JSON_NUMERIC_CHECK);		
	}

	public function obtenerINEIDistritos(){
    	/*
        * ----------------------
        * OBTIENE Y VALIDA DATOS
        * ----------------------
        */
        
		$idProvincia = $this->request->getPost('idProvincia');

        if($idProvincia == 'TOTAL')
        {
            $listadoDistritos = $this->distritoINEI->findAll();
        }
        else
        {
            $whereDistritos = array('id_prov_inei' => $idProvincia);
            $listadoDistritos = $this->distritoINEI->where($whereDistritos)->findAll();
        }
        
        $distritos=array();

		if (!empty($listadoDistritos)) {
            foreach ($listadoDistritos as $items) {
                $distritos[] = array(
					'id' => (string) $items['id_distrito'],
         	        'distrito' => (string) $items['desc_ubigeo_inei'],
                );
            }

            $resultado = array(
				'listadoDistritos' => json_encode($distritos)
			);
        } else {
			$resultado = array(
				'descripcion' => 0,
			);
		}

        /*
         * ----------------------------
         * DATOS DE AJAX PRE PROCESADOS
         * ----------------------------
         */
        echo json_encode($resultado, JSON_NUMERIC_CHECK);		
	}

    public function descargatxt($nombreTXT)
    {
        // Define el directorio igual que hiciste al crear el archivo
        //$directorio = base_url();
        $raiz = 'public/';
        // Valida el archivo recibido, eliminando caracteres no permitidos y diagonales para evitar retrocesos en ruta
        $file = (isset($nombreTXT)) ? str_replace('/', '', filter_var($nombreTXT, FILTER_SANITIZE_URL)) : '';
        
        // Separar para obtener extensión
        $partes = explode('.', $file);
        // No se especificó el archivo, no es .txt o no existe
        if($file == '' || end($partes) != 'txt' || !file_exists($raiz.$file)) {
            die('Archivo no encontrado');
        }

        header('Content-Description: File Transfer');
        header('Content-Disposition: attachment; filename='.$raiz.$file);
        header('Expires: 0');
        header('Cache-Control: must-revalidate');
        header('Pragma: public');
        header('Content-Length: ' . filesize($raiz.$file));
        header("Content-Type: text/plain");
        readfile($raiz.$file);
    }

	public function obtenerCabecera(){
    	/*
        * ----------------------
        * OBTIENE Y VALIDA DATOS
        * ----------------------
        */
        $listadoCabeceras = $this->moduloCabecera->findAll();
        
        $cabeceras=array();

		if (!empty($listadoCabeceras)) {
            foreach ($listadoCabeceras as $items) {
                $cabeceras[] = array(
					'id' => (string) $items['IDCabecera'],
         	        'cabecera' => (string) $items['Cabecera'],
                );
            }

            $resultado = array(
				'listadoCabecera' => json_encode($cabeceras)
			);
        } else {
			$resultado = array(
				'descripcion' => 0,
			);
		}

        /*
         * ----------------------------
         * DATOS DE AJAX PRE PROCESADOS
         * ----------------------------
         */
        echo json_encode($resultado, JSON_NUMERIC_CHECK);		
	}

	public function listarPerfiles(){
		if(isset($this->items['id'])){
			/* BUSCAR PERFILES */
			$wherePerfil = array(
				//'perfil.eliminacion_logica' => 1
			);

			$listaPerfiles = $this->perfilModelo
				->where($wherePerfil)
				->findAll();

			$data = array (
				'session'       => 'on',
				'titulo'        => 'Perfiles | Verisure',
				'breadcrumb'    => $this->items['modulo'],
				'listaPerfiles' => $listaPerfiles,
			);
			$data = array_merge($data, $this->items);
			return view('administracion/listar_perfiles',$data);			
		} else {
			return view('errors/500');
		}
	}

    public function listadoPerfilesSelect()
    {
        $elemento = $this->request->getPost('term');
        
        $listado = $this->perfilModelo->listadoPerfilesSelect($elemento['term']);

        $elementos=array();

        foreach ($listado as $items) {
            $elementos[] = array(
                'id' => (string) $items['IDPerfil'],
                'text' => (string) $items['Perfil']
            );
        }

        echo json_encode($elementos);
    }

    public function registrarPerfil()
    {
        $fechaActual = date("Y-m-d H:i:s");
        
        /*
        * ----------------------------
        * DATOS DE AJAX PRE PROCESADOS
        * ----------------------------
        */
        
        $msjError = TRUE;
        $perfil = ucfirst(mb_strtolower($this->request->getPost('txtPerfil')));
        $url = $this->request->getPost('txtURLBasePerfil');
        
        /*
        * ----------------------------
        * VALIDACIÓN DE REQUERIMIENTOS
        * ----------------------------
        */
        
        $error = '';
        $error .= $this->complementos->validaCampo($perfil, 'required|trim|minlength[1]|maxlength[50]', 'Perfil');
        $error .= $this->complementos->validaCampo($url, 'required|trim|minlength[1]|maxlength[200]', 'URL');

        if ($error != '') {
            $message = sprintf($this->mensajeError->msg201, $error);
            echo $this->alertas->alertaError($message);
            EXIT;
        }
        
        /*
        * ----------------------------
        * VALIDACIÓN DE EXISTENCIA
        * ----------------------------
        */

        $msgExistencia = '';
        
		$perfilExistente = $this->perfilModelo->where('Perfil', $perfil)->findAll();
        
        if(count($perfilExistente)>0)
        {
            $msgExistencia .= "<li>El perfil ya fue registrado. [CAMPO] <small>PERFIL</small></li>";
        }

        if($msgExistencia != '')
        {
            $message = sprintf($this->mensajeError->msg201, $msgExistencia);
            echo $this->alertas->alertaPeligro($message);
            EXIT;
        }


        $data = array (
            'Perfil'        => $perfil,
            'URLBase'       => $url,
            'UsuarioCrea'   => $this->items['id'],
            'FechaCrea'     => $fechaActual,    
            'Localhost'     => $this->request->getIPAddress(),
            'Estado'        => 'A'
        );

        $lastId = $this->perfilModelo->insert($data, TRUE);

		if($lastId !== FALSE){
			$message = sprintf($this->mensajeError->msg502, $perfil);
			echo $this->alertas->alertaExito($message);
			echo $this->alertas->refrescar(1);
		}
    }

	public function obtenerDatosPerfil(){
        /*
        * ----------------------------
        * DATOS DE AJAX PRE PROCESADOS
        * ----------------------------
        */

        $msjError = TRUE;
        $idPerfil = $this->request->getPost('idPerfil');

        /*
        * ----------------------------
        * VALIDACIÓN DE REQUERIMIENTOS
        * ----------------------------
        */

        $error = '';
        $error .= $this->complementos->validaCampo($idPerfil, 'required|trim|numeric', 'ID Perfil');
                
        if ($error != '') {
            $message = sprintf($this->mensajeError->msg201, $error);
            echo $this->alertas->alertaError($message);
            EXIT;
        }

		/*
        * -------------
        * BASE DE DATOS
        * -------------
        */
		
        $datosPerfil = $this->perfilModelo->where("IDPerfil", $idPerfil)->first();

        if(!empty($datosPerfil)){
			$resultado = array(
				'idperfil' => (string) $idPerfil,
				'perfil' => (string) $datosPerfil['Perfil'],
				'url' => (string) $datosPerfil['URLBase']
			);
		} else {
			$resultado = array(
				'descripcion' => 0
			);
		}

        /*
         * ----------------------------
         * DATOS DE AJAX PRE PROCESADOS
         * ----------------------------
         */
        
        echo json_encode($resultado, JSON_NUMERIC_CHECK);	
    }

	public function actualizarPerfil($idPerfil)
    {
        $fechaActual = date("Y-m-d H:i:s");
        
        /*
        * ----------------------------
        * DATOS DE AJAX PRE PROCESADOS
        * ----------------------------
        */

        $msjError = TRUE;
        $perfil = $this->request->getPost('txtPerfil');
        $url = $this->request->getPost('txtURLBasePerfil');

        /*
        * ----------------------------
        * VALIDACIÓN DE REQUERIMIENTOS
        * ----------------------------
        */

        $error = '';
        $error .= $this->complementos->validaCampo($perfil, 'required|trim|minlength[1]|maxlength[50]', 'Perfil');
        $error .= $this->complementos->validaCampo($url, 'required|trim|minlength[1]|maxlength[200]', 'URL');

        if ($error != '') {
            $message = sprintf($this->mensajeError->msg201, $error);
            echo $this->alertas->alertaError($message);
            EXIT;
        }

        /*
        * -------------
        * BASE DE DATOS
        * -------------
        */            
		
        $data = array (
            'Perfil'            => $perfil,
            'URLBase'           => $url,
            'UsuarioModifica'   => $this->items['id'],
            'FechaModifica'     => $fechaActual,    
            'Localhost'         => $this->request->getIPAddress()
        );

        $lastId = $this->perfilModelo->update([$idPerfil], $data);

        if($lastId !== FALSE){
            $message = sprintf($this->mensajeError->msg503, $perfil);
            echo $this->alertas->alertaExito($message);
			echo $this->alertas->refrescar(1);
        }		
	}

    public function modificarEstadoPerfil()
    {
        $fechaActual = date("Y-m-d H:i:s");

        /*
        * ----------------------------
        * DATOS DE AJAX PRE PROCESADOS
        * ----------------------------
        */

        $msjError = TRUE;
        $idPerfil = $this->request->getPost('idPerfil');
        $estado = $this->request->getPost('estado');

        /*
        * ----------------------------
        * VALIDACIÓN DE REQUERIMIENTOS
        * ----------------------------
        */

        $error = '';
        $error .= $this->complementos->validaCampo($idPerfil, 'required|trim|numeric', 'ID Perfil');
        $error .= $this->complementos->validaCampo($estado, 'required|trim|alpha|minlength[1]|maxlength[1]', 'Estado');

        if ($error != '') {
            $message = sprintf($this->mensajeError->msg201, $error);
            echo $this->alertas->alertaError($message);
            EXIT;
        }

        $data = array (
            'UsuarioModifica'   => $this->items['id'],
            'FechaModifica'     => $fechaActual,    
            'Localhost'         => $this->request->getIPAddress(),
            'Estado'            => $estado
        );
        $lastId = $this->perfilModelo->update([$idPerfil], $data);

		if($lastId !== FALSE){
			$message = sprintf($this->mensajeError->msg503, $idPerfil);
			echo $this->alertas->alertaExito($message);
			echo $this->alertas->refrescar(1);
		}
    }
    
	public function listarPermisosPerfil($idPerfil)
    {
		if(isset($this->items['id'])){
			/* BUSCAR PERMISOS*/
            
			$listaPermisos = $this->perfilPermisos->listarPermisosPerfil($idPerfil);
            $perfil = $this->perfilModelo->where("IDPerfil", $idPerfil)->first();
			$listaModulos = $this->modulo->findAll();

			$data = array (
				'session'           => 'on',
				'titulo'            => 'Permisos | Verisure',
				'breadcrumb'        => $this->items['modulo'],
				'breadcrumb_alias'  => 'Permisos de Perfil',
                'previo'            => 'listar-perfiles',
				'listaPermisos'     => $listaPermisos,
                'listaModulos'      => $listaModulos,
                'perfilSeleccionado'=> $perfil,
                'idPerfilSeleccionado'  => $idPerfil
			);
			$data = array_merge($data, $this->items);
			return view('administracion/listar_permisos_perfil',$data);			
		} else {
			return view('errors/500');
		}
	}

	public function listarRutasPerfil($idPerfil)
    {
		if(isset($this->items['id'])){
			/* BUSCAR PERMISOS*/
            
			$listaRutas = $this->perfilRutas->listarRutasPerfil($idPerfil);
            $perfil = $this->perfilModelo->where("IDPerfil", $idPerfil)->first();
			$listaModulos = $this->modulo->findAll();

			$data = array (
				'session'           => 'on',
				'titulo'            => 'Rutas | Verisure',
				'breadcrumb'        => $this->items['modulo'],
				'breadcrumb_alias'  => 'Rutas de Perfil',
                'previo'            => 'listar-perfiles',
				'listaRutas'        => $listaRutas,
                'listaModulos'      => $listaModulos,
                'perfilSeleccionado'=> $perfil,
                'idPerfilSeleccionado'  => $idPerfil
			);
			$data = array_merge($data, $this->items);
			return view('administracion/listar_rutas_perfil',$data);			
		} else {
			return view('errors/500');
		}
	}
	
	public function listarUsuarios(){
		if(isset($this->items['id'])){
			/* BUSCAR USUARIOS */
            /* BUSCAR PERFILES  */
            
			$data = array (
				'session'           => 'on',
				'titulo'            => 'Usuarios | Verisure',
				'breadcrumb'        => $this->items['modulo'],
				'breadcrumb_alias'  => 'Listado de usuarios'
			);
			$data = array_merge($data, $this->items);
			return view('administracion/listar_usuarios',$data);			
		} else {
			return view('errors/500');	
		}
	}

    public function grabarUsuario(){

        /*
        * ----------------------------
        * DATOS DE AJAX PRE PROCESADOS
        * ----------------------------
        */

        $msjError = TRUE;
        $perfil = $this->request->getPost('perfil');
        $usuario = $this->request->getPost('usuario');
        $nombres = $this->request->getPost('nombres');
        $apellidos = $this->request->getPost('apellidos');
        $tipoDocumento = $this->request->getPost('tipo_documento');
        $numeroDocumento = $this->request->getPost('numero_documento');
        $correoElectronico = $this->request->getPost('correo_electronico');

        /*
        * ----------------------------
        * VALIDACIÓN DE REQUERIMIENTOS
        * ----------------------------
        */

        $error = '';
        $error .= $this->complementos->validaCampo($perfil, 'required|trim|numeric|minlength[1]|maxlength[3]', 'Perfil');
        $error .= $this->complementos->validaCampo($usuario, 'required|trim|minlength[1]|maxlength[100]', 'Usuario');
        $error .= $this->complementos->validaCampo($nombres, 'required|trim|name|minlength[1]|maxlength[100]', 'Nombres');
        $error .= $this->complementos->validaCampo($apellidos, 'required|trim|name|minlength[1]|maxlength[100]', 'Apellidos');
        $error .= $this->complementos->validaCampo($tipoDocumento, 'required|trim|name|minlength[1]|maxlength[6]', 'Tipo de documento');
        $error .= $this->complementos->validaCampo($numeroDocumento, 'required|numeric|trim|minlength[8]|maxlength[12]', 'Número de documento');
        $error .= $this->complementos->validaCampo($correoElectronico, 'required|trim|email|minlength[1]|maxlength[100]', 'Correo electrónico');

        if ($error != '') {
            $message = sprintf($this->mensajeError->msg201, $error);
            echo $this->alertas->alertaError($message);
            EXIT;
        }

        $contrasena = $this->scripts->crearContrasena();
            
        $data = array (
            'usuario' => $usuario,
            'contrasena' => sha1($contrasena),
            'perfil_id' => $perfil,
            'nombres' => strtoupper($nombres),
            'apellidos' => strtoupper($apellidos),
            'tipo_documento' => $tipoDocumento,
            'numero_documento' => $numeroDocumento,
            'correo_electronico' => $correoElectronico,
            'fecha_registro' => date("Y-m-d H:i:s"),
            'eliminacion_logica' => 1,
        );
        $lastId = $this->usuarioModelo->insert($data, TRUE);

		if($lastId !== FALSE){
			$message = sprintf($this->mensajeError->msg502, $nombres.' '.$apellidos);
			echo $this->alertas->alertaExito($message);
			echo $this->alertas->refrescar(1);
		}
    }  

	public function obtenerDatosUsuario(){
        /*
        * ----------------------------
        * DATOS DE AJAX PRE PROCESADOS
        * ----------------------------
        */

        $msjError = TRUE;
        $usuario = $this->request->getPost('usuario');

        /*
        * ----------------------------
        * VALIDACIÓN DE REQUERIMIENTOS
        * ----------------------------
        */

        $error = '';
        $error .= $this->complementos->validaCampo($usuario, 'required|trim|numeric|minlength[1]|maxlength[10]', 'Usuario');
                
        if ($error != '') {
            $message = sprintf($this->mensajeError->msg201, $error);
            echo $this->alertas->alertaError($message);
            EXIT;
        }

        $whereUsuario = array (
            'usuario.id_usuario' => $usuario,
            'usuario.eliminacion_logica' => 1
        );
        $datosUsuario = $this->usuarioModelo
				->where($whereUsuario)
				->findAll();

        if(!empty($datosUsuario)){
            foreach ($datosUsuario as $items) {
                $resultado[] = array(
					'id_usuario' => (string) $items['id_usuario'],
					'usuario' => (string) $items['usuario'],
         	        'perfil_id' => (string) $items['perfil_id'],
         	        'nombres' => (string) $items['nombres'],
         	        'apellidos' => (string) $items['apellidos'],
         	        'tipo_documento' => (string) $items['tipo_documento'],
         	        'numero_documento' => (string) $items['numero_documento'],
         	        'correo_electronico' => (string) $items['correo_electronico']
                );
            }
        } else {
			$resultado = array(
				'descripcion' => 0,
			);
		}

        /*
         * ----------------------------
         * DATOS DE AJAX PRE PROCESADOS
         * ----------------------------
         */
        
        echo json_encode($resultado, JSON_NUMERIC_CHECK);	
    }

    public function actualizarUsuario($idUsuario){

        /*
        * ----------------------------
        * DATOS DE AJAX PRE PROCESADOS
        * ----------------------------
        */

        $msjError = TRUE;
        $perfil = $this->request->getPost('perfil_e');
        $usuario = $this->request->getPost('usuario_e');
        $nombres = $this->request->getPost('nombres_e');
        $apellidos = $this->request->getPost('apellidos_e');
        $tipoDocumento = $this->request->getPost('tipo_documento_e');
        $numeroDocumento = $this->request->getPost('numero_documento_e');
        $correoElectronico = $this->request->getPost('correo_electronico_e');

        /*
        * ----------------------------
        * VALIDACIÓN DE REQUERIMIENTOS
        * ----------------------------
        */

        $error = '';
        $error .= $this->complementos->validaCampo($perfil, 'required|trim|numeric|minlength[1]|maxlength[3]', 'Perfil');
        $error .= $this->complementos->validaCampo($usuario, 'required|trim|minlength[1]|maxlength[100]', 'Usuario');
        $error .= $this->complementos->validaCampo($nombres, 'required|trim|name|minlength[1]|maxlength[100]', 'Nombres');
        $error .= $this->complementos->validaCampo($apellidos, 'required|trim|name|minlength[1]|maxlength[100]', 'Apellidos');
        $error .= $this->complementos->validaCampo($tipoDocumento, 'required|trim|name|minlength[1]|maxlength[6]', 'Tipo de documento');
        $error .= $this->complementos->validaCampo($numeroDocumento, 'required|numeric|trim|minlength[8]|maxlength[12]', 'Número de documento');
        $error .= $this->complementos->validaCampo($correoElectronico, 'required|trim|email|minlength[1]|maxlength[100]', 'Correo electrónico');

        if ($error != '') {
            $message = sprintf($this->mensajeError->msg201, $error);
            echo $this->alertas->alertaError($message);
            EXIT;
        }
            
        $data = array (
            'usuario' => $usuario,
            'perfil_id' => $perfil,
            'nombres' => strtoupper($nombres),
            'apellidos' => strtoupper($apellidos),
            'tipo_documento' => $tipoDocumento,
            'numero_documento' => $numeroDocumento,
            'correo_electronico' => $correoElectronico,
            'fecha_actualizacion' => date("Y-m-d H:i:s")
        );
        $lastId = $this->usuarioModelo->update([$idUsuario], $data);

		if($lastId !== FALSE){
			$message = sprintf($this->mensajeError->msg503, $nombres.' '.$apellidos);
			echo $this->alertas->alertaExito($message);
			echo $this->alertas->refrescar(1);
		}
    }

    public function eliminarUsuario($idUsuario){
        $data = array (
            'eliminacion_logica' => 0
        );
        $lastId = $this->usuarioModelo->update([$idUsuario], $data);

		if($lastId !== FALSE){
			$message = sprintf($this->mensajeError->msg504, "usuario");
			echo $this->alertas->alertaExito($message);
			echo $this->alertas->refrescar(1);
		}
    }

    public function reactivarUsuario($idUsuario){
        $data = array (
            'eliminacion_logica' => 1
        );
        $lastId = $this->usuarioModelo->update([$idUsuario], $data);

		if($lastId !== FALSE){
			$message = sprintf($this->mensajeError->msg503, "usuario");
			echo $this->alertas->alertaExito($message);
			echo $this->alertas->refrescar(1);
		}
    }

	public function subacciones(){
    	/*
        * ----------------------
        * OBTIENE Y VALIDA DATOS
        * ----------------------
        */
        
		$accion = $this->request->getPost('accion');
        
		$whereSubAccion = array('accion_id' => $accion);				
		$listaSubAcciones = $this->subAccionModelo->where($whereSubAccion)->findAll();
        
		if (!empty($listaSubAcciones)) {
            foreach ($listaSubAcciones as $items) {
                $resultado[] = array(
					'id' => (string) $items['id_subaccion'],
         	        'subaccion' => (string) $items['subaccion'],
                );
            }
        } else {
			$resultado = array(
				'descripcion' => 0,
			);
		}

        /*
         * ----------------------------
         * DATOS DE AJAX PRE PROCESADOS
         * ----------------------------
         */
        echo json_encode($resultado, JSON_NUMERIC_CHECK);		
	}

    public function listadoDireccion()
    {
        $elemento = $this->request->getPost('term');
        
        $listado = $this->direccionModelo->listadoDireccionSelect($elemento['term']);

        $elementos=array();

        foreach ($listado as $items) {
            $elementos[] = array(
                'id' => (string) $items['id_direccion'],
                'text' => (string) $items['Direccion']." | ".$items['desc_ubigeo_inei']." - ".$items['desc_prov_inei']." | ".$items['desc_dep_inei']
            );
        }

        echo json_encode($elementos);
    }

    public function listarModulos()
    {
        if(isset($this->items['id'])){
			/* BUSCAR USUARIOS */
			$listaModulos = $this->modulo->listarModulos();
            //$listaModulos = array();
            /* BUSCAR PERFILES */
            
			$data = array (
				'session'           => 'on',
				'titulo'            => 'Módulos | Verisure',
				'breadcrumb'        => $this->items['modulo'],
				'breadcrumb_alias'  => 'Listado de módulos',
				'listaModulos'      => $listaModulos
			);
			$data = array_merge($data, $this->items);
			return view('administracion/listar_modulos',$data);			
		} else {
			return view('errors/500');	
		}
    }

    public function registrarModulo()
    {
        $fechaActual = date("Y-m-d H:i:s");
        
        /*
        * ----------------------------
        * DATOS DE AJAX PRE PROCESADOS
        * ----------------------------
        */
        
        $msjError = TRUE;
        $modulo = ucfirst(mb_strtolower($this->request->getPost('txtModulo_i')));
        $cabecera = $this->request->getPost('txtCabecera_i');
        $orden = $this->request->getPost('txtOrden_i');
        $icono = $this->request->getPost('txtIcono_i');
        $observacion = $this->request->getPost('txtObservacionModulo_i') ;
        
        /*
        * ----------------------------
        * VALIDACIÓN DE REQUERIMIENTOS
        * ----------------------------
        */
        
        $error = '';
        $error .= $this->complementos->validaCampo($modulo, 'required|trim|minlength[1]|maxlength[50]', 'Módulo');
        $error .= $this->complementos->validaCampo($cabecera, 'required|trim|numeric', 'Cabecera');
        $error .= $this->complementos->validaCampo($orden, 'required|trim|numeric', 'Orden');
        $error .= $this->complementos->validaCampo($icono, 'required|trim', 'Ícono');
        $error .= $this->complementos->validaCampo($observacion, 'required|trim', 'Observación');

        if ($error != '') {
            $message = sprintf($this->mensajeError->msg201, $error);
            echo $this->alertas->alertaError($message);
            EXIT;
        }
        
        /*
        * ----------------------------
        * VALIDACIÓN DE EXISTENCIA
        * ----------------------------
        */

        $msgExistencia = '';
        
		$moduloExistente = $this->modulo->where('Modulo', $modulo)->findAll();
        
        if(count($moduloExistente)>0)
        {
            $msgExistencia .= "<li>El módulo ya fue registrado. [CAMPO] <small>MÓDULO</small></li>";
        }

        if($msgExistencia != '')
        {
            $message = sprintf($this->mensajeError->msg201, $msgExistencia);
            echo $this->alertas->alertaPeligro($message);
            EXIT;
        }


        $data = array (
            'IDCabecera'    => $cabecera,
            'Modulo'        => $modulo,
            'Orden'         => $orden,
            'Icono'         => $icono,
            'Observacion'   => $observacion,
            'UsuarioCrea'   => $this->items['id'],
            'FechaCrea'     => $fechaActual,    
            'Localhost'     => $this->request->getIPAddress(),
            'Estado'        => 'A'
        );

        $lastId = $this->modulo->insert($data, TRUE);

		if($lastId !== FALSE){
			$message = sprintf($this->mensajeError->msg502, $modulo);
			echo $this->alertas->alertaExito($message);
			echo $this->alertas->refrescar(1);
		}
    }

    public function modificarModulo($idModulo){

        $fechaActual = date("Y-m-d H:i:s");

        /*
        * ----------------------------
        * DATOS DE AJAX PRE PROCESADOS
        * ----------------------------
        */

        $msjError = TRUE;
        $modulo = ucfirst(mb_strtolower($this->request->getPost('txtModulo_i')));
        $cabecera = $this->request->getPost('txtCabecera_i');
        $orden = $this->request->getPost('txtOrden_i');
        $icono = $this->request->getPost('txtIcono_i');
        $observacion = $this->request->getPost('txtObservacionModulo_i');

        /*
        * ----------------------------
        * VALIDACIÓN DE REQUERIMIENTOS
        * ----------------------------
        */

        $error = '';
        $error .= $this->complementos->validaCampo($modulo, 'required|trim', 'Modulo');
        $error .= $this->complementos->validaCampo($cabecera, 'required|trim|numeric', 'Cabecera');
        $error .= $this->complementos->validaCampo($orden, 'required|trim|numeric', 'Orden');
        $error .= $this->complementos->validaCampo($icono, 'required|trim', 'Ícono');
        $error .= $this->complementos->validaCampo($observacion, 'required|trim', 'Observación');

        if ($error != '') {
            $message = sprintf($this->mensajeError->msg201, $error);
            echo $this->alertas->alertaError($message);
            EXIT;
        }

        $data = array (
            'IDCabecera'        => $cabecera,
            'Modulo'            => $modulo,
            'Orden'             => $orden,
            'Icono'             => $icono,
            'Observacion'       => $observacion,
            'UsuarioModifica'   => $this->items['id'],
            'FechaModifica'     => $fechaActual,    
            'Localhost'         => $this->request->getIPAddress()
        );
        $lastId = $this->modulo->update([$idModulo], $data);

		if($lastId !== FALSE){
			$message = sprintf($this->mensajeError->msg503, $idModulo);
			echo $this->alertas->alertaExito($message);
			echo $this->alertas->refrescar(1);
		}
    }
    
    public function modificarEstadoModulo()
    {
        $fechaActual = date("Y-m-d H:i:s");

        /*
        * ----------------------------
        * DATOS DE AJAX PRE PROCESADOS
        * ----------------------------
        */

        $msjError = TRUE;
        $idModulo = $this->request->getPost('idModulo');
        $estado = $this->request->getPost('estado');

        /*
        * ----------------------------
        * VALIDACIÓN DE REQUERIMIENTOS
        * ----------------------------
        */

        $error = '';
        $error .= $this->complementos->validaCampo($idModulo, 'required|trim|numeric', 'ID Módulo');
        $error .= $this->complementos->validaCampo($estado, 'required|trim|alpha|minlength[1]|maxlength[1]', 'Estado');

        if ($error != '') {
            $message = sprintf($this->mensajeError->msg201, $error);
            echo $this->alertas->alertaError($message);
            EXIT;
        }

        $data = array (
            'UsuarioModifica'   => $this->items['id'],
            'FechaModifica'     => $fechaActual,    
            'Localhost'         => $this->request->getIPAddress(),
            'Estado'            => $estado
        );
        $lastId = $this->modulo->update([$idModulo], $data);

		if($lastId !== FALSE){
			$message = sprintf($this->mensajeError->msg503, $idModulo);
			echo $this->alertas->alertaExito($message);
			echo $this->alertas->refrescar(1);
		}
    }
    
	public function obtenerDatosModulo() {

		/*
        * ----------------------------
        * DATOS DE AJAX PRE PROCESADOS
        * ----------------------------
		*/

		$msjError = TRUE;
		$idModulo = $this->request->getPost('idModulo');

		/*
        * ----------------------------
        * VALIDACIÓN DE REQUERIMIENTOS
        * ----------------------------
        */

		$error = '';
		$error .= $this->complementos->validaCampo($idModulo, 'trim', 'ID');

		if ($error != '') {
			$message = sprintf($this->mensajeError->msg201, $error);
			echo $this->alertas->alertaError($message);
			exit;
		}

		/*
        * -------------
        * BASE DE DATOS
        * -------------
        */
		
		$datosModulo = $this->modulo->where('IDModulo', $idModulo)->first();

		if(!empty($datosModulo)){
			$resultado = array(
				'idmodulo' => (string) $idModulo,
				'cabecera' => (string) $datosModulo['IDCabecera'],
				'modulo' => (string) $datosModulo['Modulo'],
				'orden' => (string) $datosModulo['Orden'],
				'icono' => (string) $datosModulo['Icono'],
				'observacion' => (string) $datosModulo['Observacion']
			);
		} else {
			$resultado = array(
				'descripcion' => 0
			);
		}

		/*
         * ----------------------------
         * DATOS DE AJAX PRE PROCESADOS
         * ----------------------------
         */

		echo json_encode($resultado);		
	}
    
    public function listarRutasModulo($idModulo)
    {
        if(isset($this->items['id'])){
			/* BUSCAR RUTAS */
			$listaRutas = $this->moduloRutas->where("IDModulo", $idModulo)->findAll();
            $modulo = $this->modulo->where('IDModulo', $idModulo)->first();

			$data = array (
				'session'           => 'on',
				'titulo'            => 'Rutas | Verisure',
				'breadcrumb'        => $this->items['modulo'],
				'breadcrumb_alias'  => 'Listado de rutas',
                'previo'            => 'listar-modulos',
				'listaRutas'        => $listaRutas,
                'moduloSeleccionado'=> $modulo,
                'idModulo'          => $idModulo
			);
			$data = array_merge($data, $this->items);
			return view('administracion/listar_rutas_modulos',$data);			
		} else {
			return view('errors/500');	
		}
    }

    public function guardarRutaModulo($idModulo)
    {
        $fechaActual = date("Y-m-d H:i:s");
        
        /*
        * ----------------------------
        * DATOS DE AJAX PRE PROCESADOS
        * ----------------------------
        */
        
        $msjError = TRUE;
        $ruta = $this->request->getPost('txtRutaModulo') ;
        $tipo = $this->request->getPost('txtTipoRuta_i') ;
        $raiz = $this->request->getPost('chkRutaRaiz') ;
        
        /*
        * ----------------------------
        * VALIDACIÓN DE REQUERIMIENTOS
        * ----------------------------
        */
        
        $error = '';
        $error .= $this->complementos->validaCampo($ruta, 'required|trim', 'Ruta');
        $error .= $this->complementos->validaCampo($tipo, 'required|trim', 'Tipo de Ruta');

        if ($error != '') {
            $message = sprintf($this->mensajeError->msg201, $error);
            echo $this->alertas->alertaError($message);
            EXIT;
        }
        
        /*
        * ----------------------------
        * VALIDACIÓN DE EXISTENCIA
        * ----------------------------
        */

        $msgExistencia = '';
        
		$rutaExistente = $this->moduloRutas->where('Ruta', $ruta)->findAll();
        
        if(count($rutaExistente)>0)
        {
            $msgExistencia .= "<li>La ruta ya fue registrada. [CAMPO] <small>RUTA</small></li>";
        }

        if($msgExistencia != '')
        {
            $message = sprintf($this->mensajeError->msg201, $msgExistencia);
            echo $this->alertas->alertaPeligro($message);
            EXIT;
        }


        $data = array (
            'IDModulo'  => $idModulo, 
            'Ruta'      => $ruta,
            'TipoRuta'  => $tipo,
            'RaizModulo'=> (isset($raiz) ? 'Y' : 'N'),
            'Estado'    => 'A'
        );

        $lastId = $this->moduloRutas->insert($data, TRUE);

		if($lastId !== FALSE){
			$message = sprintf($this->mensajeError->msg502, $ruta);
			echo $this->alertas->alertaExito($message);
			echo $this->alertas->refrescar(1);
		}
    }

	public function obtenerDatosRutaModulo() 
    {

		/*
        * ----------------------------
        * DATOS DE AJAX PRE PROCESADOS
        * ----------------------------
		*/

		$msjError = TRUE;
		$idRutaModulo = $this->request->getPost('idRutaModulo');

		/*
        * ----------------------------
        * VALIDACIÓN DE REQUERIMIENTOS
        * ----------------------------
        */

		$error = '';
		$error .= $this->complementos->validaCampo($idRutaModulo, 'trim', 'ID');

		if ($error != '') {
			$message = sprintf($this->mensajeError->msg201, $error);
			echo $this->alertas->alertaError($message);
			exit;
		}

		/*
        * -------------
        * BASE DE DATOS
        * -------------
        */
		
		$datosRuta = $this->moduloRutas->where('IDRuta', $idRutaModulo)->first();

		if(!empty($datosRuta)){
			$resultado = array(
				'idruta' => (string) $idRutaModulo,
				'ruta' => (string) $datosRuta['Ruta'],
				'tipo' => (string) $datosRuta['TipoRuta'],
				'raiz' => (string) $datosRuta['RaizModulo']
			);
		} else {
			$resultado = array(
				'descripcion' => 0
			);
		}

		/*
         * ----------------------------
         * DATOS DE AJAX PRE PROCESADOS
         * ----------------------------
         */

		echo json_encode($resultado);		
	}

    public function modificarRutaModulo($idRutaModulo){

        $fechaActual = date("Y-m-d H:i:s");

        /*
        * ----------------------------
        * DATOS DE AJAX PRE PROCESADOS
        * ----------------------------
        */

        $msjError = TRUE;
        $ruta = $this->request->getPost('txtRutaModulo');
        $tipo = $this->request->getPost('txtTipoRuta_i');
        $raiz = $this->request->getPost('chkRutaRaiz');

        /*
        * ----------------------------
        * VALIDACIÓN DE REQUERIMIENTOS
        * ----------------------------
        */

        $error = '';
        $error .= $this->complementos->validaCampo($ruta, 'required|trim', 'Ruta');
        $error .= $this->complementos->validaCampo($tipo, 'required|trim', 'Tipo de Ruta');

        if ($error != '') {
            $message = sprintf($this->mensajeError->msg201, $error);
            echo $this->alertas->alertaError($message);
            EXIT;
        }

        $data = array (
            'Ruta'              => $ruta,
            'TipoRuta'          => $tipo,
            'RaizModulo'        => (isset($raiz) ? 'Y' : 'N')
        );
        $lastId = $this->moduloRutas->update([$idRutaModulo], $data);

		if($lastId !== FALSE){
			$message = sprintf($this->mensajeError->msg503, $idRutaModulo);
			echo $this->alertas->alertaExito($message);
			echo $this->alertas->refrescar(1);
		}
    }

    public function modificarEstadoRutaModulo(){

        $fechaActual = date("Y-m-d H:i:s");

        /*
        * ----------------------------
        * DATOS DE AJAX PRE PROCESADOS
        * ----------------------------
        */

        $msjError = TRUE;
        $idRutaModulo = $this->request->getPost('idRutaModulo');
        $estado = $this->request->getPost('estado');

        /*
        * ----------------------------
        * VALIDACIÓN DE REQUERIMIENTOS
        * ----------------------------
        */

        $error = '';
        $error .= $this->complementos->validaCampo($idRutaModulo, 'required|trim|numeric', 'ID Ruta Módulo');
        $error .= $this->complementos->validaCampo($estado, 'required|trim|alpha|minlength[1]|maxlength[1]', 'Estado');

        if ($error != '') {
            $message = sprintf($this->mensajeError->msg201, $error);
            echo $this->alertas->alertaError($message);
            EXIT;
        }

        $data = array (
            'Estado'            => $estado
        );
        $lastId = $this->moduloRutas->update([$idRutaModulo], $data);

		if($lastId !== FALSE){
			$message = sprintf($this->mensajeError->msg503, $idRutaModulo);
			echo $this->alertas->alertaExito($message);
			echo $this->alertas->refrescar(1);
		}
    }

    public function listarAccionesModulo($idModulo)
    {
        if(isset($this->items['id'])){
			/* BUSCAR ACCIONES */
			$listaAcciones = $this->moduloAcciones->where("IDModulo", $idModulo)->findAll();
            
			$data = array (
				'session'           => 'on',
				'titulo'            => 'Acciones | Verisure',
				'breadcrumb'        => $this->items['modulo'],
				'breadcrumb_alias'  => 'Listado de acciones',
                'previo'            => 'listar-modulos',
				'listaAcciones'     => $listaAcciones,
                'idModulo'          => $idModulo
			);
			$data = array_merge($data, $this->items);
			return view('administracion/listar_acciones_modulos',$data);			
		} else {
			return view('errors/500');	
		}
    }
    
    public function guardarAccionModulo($idModulo)
    {
        $fechaActual = date("Y-m-d H:i:s");
        
        /*
        * ----------------------------
        * DATOS DE AJAX PRE PROCESADOS
        * ----------------------------
        */
        
        $msjError = TRUE;
        $codigo = $this->request->getPost('txtCodigoAccionModulo') ;
        $accion = $this->request->getPost('txtAccionModulo') ;
        
        /*
        * ----------------------------
        * VALIDACIÓN DE REQUERIMIENTOS
        * ----------------------------
        */
        
        $error = '';
        $error .= $this->complementos->validaCampo($accion, 'required|trim', 'Acción');
        $error .= $this->complementos->validaCampo($codigo, 'required|trim|minlength[6]|maxlength[6]', 'Código');

        if ($error != '') {
            $message = sprintf($this->mensajeError->msg201, $error);
            echo $this->alertas->alertaError($message);
            EXIT;
        }
        
        /*
        * ----------------------------
        * VALIDACIÓN DE EXISTENCIA
        * ----------------------------
        */

        $msgExistencia = '';
        
		$accionExistente = $this->moduloAcciones->where('Accion', $accion)->findAll();
		$codigoExistente = $this->moduloAcciones->where('Codigo', $codigo)->findAll();
        
        if(count($accionExistente)>0)
        {
            $msgExistencia .= "<li>La acción ya fue registrada. [CAMPO] <small>ACCIÓN</small></li>";
        }

        if(count($codigoExistente)>0)
        {
            $msgExistencia .= "<li>El código ya fue registrada. [CAMPO] <small>CÓDIGO</small></li>";
        }

        if($msgExistencia != '')
        {
            $message = sprintf($this->mensajeError->msg201, $msgExistencia);
            echo $this->alertas->alertaPeligro($message);
            EXIT;
        }


        $data = array (
            'IDModulo'  => $idModulo, 
            'Codigo'    => $codigo,
            'Accion'    => $accion,
            'Estado'    => 'A'
        );

        $lastId = $this->moduloAcciones->insert($data, TRUE);

		if($lastId !== FALSE){
			$message = sprintf($this->mensajeError->msg502, $accion);
			echo $this->alertas->alertaExito($message);
			echo $this->alertas->refrescar(1);
		}
    }

    public function guardarPermisosPerfil($idPerfil)
    {
        $fechaActual = date("Y-m-d H:i:s");
        
        /*
        * ----------------------------
        * DATOS DE AJAX PRE PROCESADOS
        * ----------------------------
        */
        
        $msjError = TRUE;
        $permisos = $this->request->getPost('permisos') ;
        
        /*
        * ----------------------------
        * VALIDACIÓN DE REQUERIMIENTOS
        * ----------------------------
        */

        $error = '';

        if(isset($permisos) && empty($permisos))
        {
            $error .= "<li>No hay permisos seleccionados.</li>";
        }

        if ($error != '') {
            $message = sprintf($this->mensajeError->msg201, $error);
            echo $this->alertas->alertaError($message);
            EXIT;
        }
        
        /*
        * ----------------------------
        * VALIDACIÓN DE EXISTENCIA
        * ----------------------------
        */

        /*
        * ----------------------------
        * LIMPIAR EXISTENCIA PREVIA
        * ----------------------------
        */

        $listadoPermisosPrev = $this->perfilPermisos->where("IDPerfil", $idPerfil)->findAll();
//var_dump($listadoPermisosPrev);exit;
        if(count($listadoPermisosPrev) > 0)
        {
            $this->perfilPermisos->delete($idPerfil);
        }

        /*
        * ----------------------------
        * GENERAR NUEVA DATA
        * ----------------------------
        */

        $dataPermisos = array();

        foreach($permisos as $key => $value)
        {
            $dataPermisos[] = array (
                'IDPerfil'          => $idPerfil, 
                'IDAccionModulo'    => $key,
                'UsuarioRegistra'   => $this->items['id'],
                'FechaRegistra'     => $fechaActual
            );
        }
        $lastId = $this->perfilPermisos->insertBatch($dataPermisos, TRUE);

		if($lastId !== FALSE){
			$message = sprintf($this->mensajeError->msg502, $idPerfil);
			echo $this->alertas->alertaExito($message);
			echo $this->alertas->refrescar(1);
		}
    }
    
    public function guardarRutasPerfil($idPerfil)
    {
        $fechaActual = date("Y-m-d H:i:s");
        
        /*
        * ----------------------------
        * DATOS DE AJAX PRE PROCESADOS
        * ----------------------------
        */
        
        $msjError = TRUE;
        $rutas = $this->request->getPost('rutas') ;
        
        /*
        * ----------------------------
        * VALIDACIÓN DE REQUERIMIENTOS
        * ----------------------------
        */

        $error = '';

        if(isset($rutas) && empty($rutas))
        {
            $error .= "<li>No hay rutas seleccionados.</li>";
        }

        if ($error != '') {
            $message = sprintf($this->mensajeError->msg201, $error);
            echo $this->alertas->alertaError($message);
            EXIT;
        }
        
        /*
        * ----------------------------
        * VALIDACIÓN DE EXISTENCIA
        * ----------------------------
        */

        /*
        * ----------------------------
        * LIMPIAR EXISTENCIA PREVIA
        * ----------------------------
        */

        $listadoRutasPrev = $this->perfilRutas->where("IDPerfil", $idPerfil)->findAll();

        if(count($listadoRutasPrev) > 0)
        {
            $this->perfilRutas->delete($idPerfil);
        }

        /*
        * ----------------------------
        * GENERAR NUEVA DATA
        * ----------------------------
        */

        $dataRutas = array();

        foreach($rutas as $key => $value)
        {
            $dataRutas[] = array (
                'IDPerfil'          => $idPerfil, 
                'IDRuta'            => $key,
                'UsuarioRegistra'   => $this->items['id'],
                'FechaRegistra'     => $fechaActual
            );
        }
        $lastId = $this->perfilRutas->insertBatch($dataRutas, TRUE);

		if($lastId !== FALSE){
			$message = sprintf($this->mensajeError->msg502, $idPerfil);
			echo $this->alertas->alertaExito($message);
			echo $this->alertas->refrescar(1);
		}
    }
    
    
    public function listarSistemas(){
		if(isset($this->items['id'])){

			/* ACCIONES */
			$listadoSistemas = $this->sistemas->findAll();
			
			$data = array (
				'session'           => 'on',
				'titulo'            => 'Listado de Sistemas | Verisure',
				'breadcrumb'        => $this->items['modulo'],
				'breadcrumb_alias'  => 'Sistemas',
				'listadoSistemas'   => $listadoSistemas
			);
			$data = array_merge($data, $this->items);
			return view('administracion/listar_sistemas',$data);			
		} else {
			return view('errors/500');
		}
	}
	
	public function registrarSistema()
	{
        $fechaActual = date("Y-m-d H:i:s");
        
        /*
        * ----------------------------
        * DATOS DE AJAX PRE PROCESADOS
        * ----------------------------
        */
        
        $msjError = TRUE;
        $sistema = $this->request->getPost('txtSistema_i') ;
        $raiz = $this->request->getPost('txtURLRaiz_i');
        $observacion = $this->request->getPost('txtObservacion_i');
        
        /*
        * ----------------------------
        * VALIDACIÓN DE REQUERIMIENTOS
        * ----------------------------
        */
        
        $error = '';
        $error .= $this->complementos->validaCampo($sistema, 'required|trim|minlength[1]|maxlength[50]', 'Nombre Sistema');
        $error .= $this->complementos->validaCampo($raiz, 'required|trim|minlength[1]|maxlength[200]', 'URL Raiz');

        if ($error != '') {
            $message = sprintf($this->mensajeError->msg201, $error);
            echo $this->alertas->alertaError($message);
            EXIT;
        }
        
        /*
        * ----------------------------
        * VALIDACIÓN DE EXISTENCIA
        * ----------------------------
        */

        $msgExistencia = '';
        
		$raizExistente = $this->sistemas->where('Raiz', $raiz)->findAll();
        
		$sistemaExistente = $this->sistemas->where('Sistema', $sistema)->findAll();
        
        if(count($sistemaExistente)>0)
        {
            $msgExistencia .= "<li>El sistema ya fue registrado. [CAMPO] <small>SISTEMA</small></li>";
        }

        if(count($raizExistente)>0)
        {
            $msgExistencia .= "<li>La URL Raiz ya fue registrado. [CAMPO] <small>URL RAÍZ</small></li>";
        }

        if($msgExistencia != '')
        {
            $message = sprintf($this->mensajeError->msg201, $msgExistencia);
            echo $this->alertas->alertaPeligro($message);
            EXIT;
        }


        $data = array (
            'Sistema'       => $sistema,
            'Raiz'          => $raiz,
            'Observacion'   => $observacion,
            'UsuarioCrea'   => $this->items['id'],
            'FechaCrea'     => $fechaActual,    
            'Localhost'     => $this->request->getIPAddress(),
            'UsuarioEstado' => $this->items['id'],
            'FechaEstado'   => $fechaActual,    
            'MotivoEstado'  => "Creación de sistema.",    
            'Estado'        => 'A'
        );

        $lastId = $this->sistemas->insert($data, TRUE);

		if($lastId !== FALSE){
			$message = sprintf($this->mensajeError->msg502, $sistema);
			echo $this->alertas->alertaExito($message);
			echo $this->alertas->refrescar(1);
		}
	}
	
	public function obtenerDatosSistema()
	{
		/*
        * ----------------------------
        * DATOS DE AJAX PRE PROCESADOS
        * ----------------------------
		*/

		$msjError = TRUE;
		$idSistema = $this->request->getPost('idSistema');

		/*
        * ----------------------------
        * VALIDACIÓN DE REQUERIMIENTOS
        * ----------------------------
        */

		$error = '';
		$error .= $this->complementos->validaCampo($idSistema, 'trim', 'ID');

		if ($error != '') {
			$message = sprintf($this->mensajeError->msg201, $error);
			echo $this->alertas->alertaError($message);
			exit;
		}

		/*
        * -------------
        * BASE DE DATOS
        * -------------
        */
		
		$datosSistema = $this->sistemas->where('IDSistema', $idSistema)->first();

		if(!empty($datosSistema)){
			$resultado = array(
				'idsistema'     => (string) $idSistema,
				'sistema'       => (string) $datosSistema['Sistema'],
				'raiz'          => (string) $datosSistema['Raiz'],
				'observacion'   => (string) $datosSistema['Observacion']
			);
		} else {
			$resultado = array(
				'descripcion' => 0
			);
		}

		/*
         * ----------------------------
         * DATOS DE AJAX PRE PROCESADOS
         * ----------------------------
         */

		echo json_encode($resultado);		
	}
	
	public function actualizarSistema($idSistema)
	{
        $fechaActual = date("Y-m-d H:i:s");
        
        /*
        * ----------------------------
        * DATOS DE AJAX PRE PROCESADOS
        * ----------------------------
        */
        
        $sistema = $this->request->getPost('txtSistema_i') ;
        $raiz = $this->request->getPost('txtURLRaiz_i');
        $observacion = $this->request->getPost('txtObservacion_i');
        
        /*
        * ----------------------------
        * VALIDACIÓN DE REQUERIMIENTOS
        * ----------------------------
        */
        
        $error = '';
        $error .= $this->complementos->validaCampo($sistema, 'required|trim|minlength[1]|maxlength[50]', 'Nombre Sistema');
        $error .= $this->complementos->validaCampo($raiz, 'required|trim|minlength[1]|maxlength[200]', 'URL Raiz');

        if ($error != '') {
            $message = sprintf($this->mensajeError->msg201, $error);
            echo $this->alertas->alertaError($message);
            EXIT;
        }
        
        /*
        * -------------
        * BASE DE DATOS
        * -------------
        */            
		
        $data = array (
            'Sistema'           => $sistema,
            'Raiz'              => $raiz,
            'Observacion'       => $observacion,
            'UsuarioModifica'   => $this->items['id'],
            'FechaModifica'     => $fechaActual,    
            'Localhost'         => $this->request->getIPAddress()
        );

        $lastId = $this->sistemas->update([$idSistema], $data);

        if($lastId !== FALSE){
            $message = sprintf($this->mensajeError->msg503, $sistema);
            echo $this->alertas->alertaExito($message);
			echo $this->alertas->refrescar(1);
        }		
	}
	
    public function modificarEstadoSistema()
    {
        $fechaActual = date("Y-m-d H:i:s");

        /*
        * ----------------------------
        * DATOS DE AJAX PRE PROCESADOS
        * ----------------------------
        */

        $msjError = TRUE;
        $idSistema = $this->request->getPost('idSistema');
        $estado = $this->request->getPost('estado');
        $motivo = $this->request->getPost('motivo');

        /*
        * ----------------------------
        * VALIDACIÓN DE REQUERIMIENTOS
        * ----------------------------
        */

        $error = '';
        $error .= $this->complementos->validaCampo($idSistema, 'required|trim|numeric', 'ID Sistema');
        $error .= $this->complementos->validaCampo($estado, 'required|trim|alpha|minlength[1]|maxlength[1]', 'Estado');
        $error .= $this->complementos->validaCampo($motivo, 'required|trim', 'Motivo');

        if ($error != '') {
            $message = sprintf($this->mensajeError->msg201, $error);
            echo $this->alertas->alertaError($message);
            EXIT;
        }

        $data = array (
            'UsuarioModifica'   => $this->items['id'],
            'FechaModifica'     => $fechaActual,    
            'Localhost'         => $this->request->getIPAddress(),
            'UsuarioEstado'     => $this->items['id'],
            'FechaEstado'       => $fechaActual,
            'MotivoEstado'      => $motivo,
            'Estado'            => $estado
        );
        
        $lastId = $this->sistemas->update([$idSistema], $data);

		if($lastId !== FALSE){
			$message = sprintf($this->mensajeError->msg503, $idSistema);
			echo $this->alertas->alertaExito($message);
			echo $this->alertas->refrescar(1);
		}
    }
    
    public function listadoAreas()
    {
        // $elemento = $this->request->getPost('term');
        
        // $listado = $this->areas->listadoAreasSelect($elemento['term']);
        
        $idDepartamento = $this->request->getPost("idDepartamento");
        
        if($idDepartamento == 'TOTAL')
        {
            $listado = $this->areas->join('tbldepartamento', 'tblarea.IdDpto = tbldepartamento.IdDpto', "left")->where("tblarea.Estado", "A")->findAll();
        }
        else
        {
            $listado = $this->areas->listadoAreasXDepartamento($idDepartamento);
        }

        $elementos=array();

		if (!empty($listado)) {
            foreach ($listado as $items) {
                $elementos[] = array(
					'id' => (string) $items['IdArea'],
                    'text' => (string) $items['LN']." - ".$items['NombreArea']." | ".$items['CC']." - ".$items['NombreDpto']
                );
            }

            $resultado = array(
				'listadoAreas' => json_encode($elementos)
			);
        } else {
			$resultado = array(
				'descripcion' => 0,
			);
		}

        /*
         * ----------------------------
         * DATOS DE AJAX PRE PROCESADOS
         * ----------------------------
         */
        echo json_encode($resultado, JSON_NUMERIC_CHECK);		

    }

    public function listarAreas()
    {
        if(isset($this->items['id'])){

			/* ACCIONES */
			$listadoAreas = $this->areas->listadoAreas();
			
			$data = array (
				'session'           => 'on',
				'titulo'            => 'Listado de Áreas | Verisure',
				'breadcrumb'        => $this->items['modulo'],
				'breadcrumb_alias'  => 'Áreas',
				'listadoAreas'      => $listadoAreas
			);
			$data = array_merge($data, $this->items);
			return view('administracion/listar_areas',$data);			
		} else {
			return view('errors/500');
		}
    }
    
	public function registrarArea()
	{
        $fechaActual = date("Y-m-d H:i:s");
        
        /*
        * ----------------------------
        * DATOS DE AJAX PRE PROCESADOS
        * ----------------------------
        */
        
        $area = $this->request->getPost('txtArea_i') ;
        $ln = $this->request->getPost('txtLN_i');
        $idDpto = $this->request->getPost('txtDptoArea_i');
        $responsable = $this->request->getPost('txtResponsableArea_i');
        
        /*
        * ----------------------------
        * VALIDACIÓN DE REQUERIMIENTOS
        * ----------------------------
        */
        
        $error = '';
        $error .= $this->complementos->validaCampo($area, 'required|trim|minlength[1]|maxlength[100]', 'Área');
        $error .= $this->complementos->validaCampo($ln, 'required|trim|numeric', 'CC');
        $error .= $this->complementos->validaCampo($idDpto, 'required|trim|numeric', 'Departamento');
        //$error .= $this->complementos->validaCampo($responsable, 'required|trim|numeric', 'Responsable');

        if ($error != '') {
            $message = sprintf($this->mensajeError->msg201, $error);
            echo $this->alertas->alertaError($message);
            EXIT;
        }
        
        /*
        * ----------------------------
        * VALIDACIÓN DE EXISTENCIA
        * ----------------------------
        */

        $msgExistencia = '';
        
		$areaExistente = $this->areas->where('NombreArea', $area)->findAll();
        
        if(count($areaExistente)>0)
        {
            $msgExistencia .= "<li>El área ya fue registrado. [CAMPO] <small>ÁREA</small></li>";
        }

        if($msgExistencia != '')
        {
            $message = sprintf($this->mensajeError->msg201, $msgExistencia);
            echo $this->alertas->alertaPeligro($message);
            EXIT;
        }


        $data = array (
            'IdDpto'                => $idDpto,
            'LN'                    => $ln,
            'NombreArea'            => $area,
            'UsuarioResponsable'    => $responsable,
            'UsuarioCrea'           => $this->items['id'],
            'FechaCrea'             => $fechaActual,    
            'Localhost'             => $this->request->getIPAddress(),
            'UsuarioEstado'         => $this->items['id'],
            'FechaEstado'           => $fechaActual,    
            'MotivoEstado'          => "Creación de área.",    
            'Estado'                => 'A'
        );

        $lastId = $this->areas->insert($data, TRUE);

		if($lastId !== FALSE){
			$message = sprintf($this->mensajeError->msg502, $area);
			echo $this->alertas->alertaExito($message);
			echo $this->alertas->refrescar(1);
		}
	}
	
	public function obtenerDatosArea()
	{
		/*
        * ----------------------------
        * DATOS DE AJAX PRE PROCESADOS
        * ----------------------------
		*/

		$msjError = TRUE;
		$idArea = $this->request->getPost('idArea');

		/*
        * ----------------------------
        * VALIDACIÓN DE REQUERIMIENTOS
        * ----------------------------
        */

		$error = '';
		$error .= $this->complementos->validaCampo($idArea, 'trim', 'ID');

		if ($error != '') {
			$message = sprintf($this->mensajeError->msg201, $error);
			echo $this->alertas->alertaError($message);
			exit;
		}

		/*
        * -------------
        * BASE DE DATOS
        * -------------
        */
		
		$datosArea = $this->areas->where('IdArea', $idArea)->first();

		if(!empty($datosArea)){
			$resultado = array(
				'idarea'        => (string) $idArea,
				'iddepartamento'=> (string) $datosArea['IdDpto'],
				'responsable'   => (string) $datosArea['UsuarioResponsable'],
				'area'          => (string) $datosArea['NombreArea'],
				'ln'            => (string) $datosArea['LN']
			);
		} else {
			$resultado = array(
				'descripcion' => 0
			);
		}

		/*
         * ----------------------------
         * DATOS DE AJAX PRE PROCESADOS
         * ----------------------------
         */

		echo json_encode($resultado);		
	}
	
	public function actualizarArea($idArea)
	{
        $fechaActual = date("Y-m-d H:i:s");
        
        /*
        * ----------------------------
        * DATOS DE AJAX PRE PROCESADOS
        * ----------------------------
        */
        
        $area = $this->request->getPost('txtArea_i') ;
        $ln = $this->request->getPost('txtLN_i');
        $idDpto = $this->request->getPost('txtDptoArea_i');
        $responsable = $this->request->getPost('txtResponsableArea_i');
        
        /*
        * ----------------------------
        * VALIDACIÓN DE REQUERIMIENTOS
        * ----------------------------
        */
        
        $error = '';
        $error .= $this->complementos->validaCampo($area, 'required|trim|minlength[1]|maxlength[100]', 'Área');
        $error .= $this->complementos->validaCampo($ln, 'required|trim|numeric', 'CC');
        $error .= $this->complementos->validaCampo($idDpto, 'required|trim|numeric', 'Departamento');
        //$error .= $this->complementos->validaCampo($responsable, 'required|trim|numeric', 'Responsable');

        if ($error != '') {
            $message = sprintf($this->mensajeError->msg201, $error);
            echo $this->alertas->alertaError($message);
            EXIT;
        }
        
        /*
        * -------------
        * BASE DE DATOS
        * -------------
        */            
		
        $data = array (
            'IdDpto'            => $idDpto,
            'LN'                => $ln,
            'NombreArea'        => $area,
            'UsuarioResponsable'=> $responsable,
            'UsuarioModifica'   => $this->items['id'],
            'FechaModifica'     => $fechaActual,    
            'Localhost'         => $this->request->getIPAddress()
        );

        $lastId = $this->areas->update([$idArea], $data);

        if($lastId !== FALSE){
            $message = sprintf($this->mensajeError->msg503, $area);
            echo $this->alertas->alertaExito($message);
			//echo $this->alertas->refrescar(1);
        }		
	}
	
    public function modificarEstadoArea()
    {
        $fechaActual = date("Y-m-d H:i:s");

        /*
        * ----------------------------
        * DATOS DE AJAX PRE PROCESADOS
        * ----------------------------
        */

        $msjError = TRUE;
        $idArea = $this->request->getPost('idArea');
        $estado = $this->request->getPost('estado');
        $motivo = $this->request->getPost('motivo');

        /*
        * ----------------------------
        * VALIDACIÓN DE REQUERIMIENTOS
        * ----------------------------
        */

        $error = '';
        $error .= $this->complementos->validaCampo($idArea, 'required|trim|numeric', 'ID Área');
        $error .= $this->complementos->validaCampo($estado, 'required|trim|alpha|minlength[1]|maxlength[1]', 'Estado');
        $error .= $this->complementos->validaCampo($motivo, 'required|trim', 'Motivo');

        if ($error != '') {
            $message = sprintf($this->mensajeError->msg201, $error);
            echo $this->alertas->alertaError($message);
            EXIT;
        }

        $data = array (
            'UsuarioModifica'   => $this->items['id'],
            'FechaModifica'     => $fechaActual,    
            'Localhost'         => $this->request->getIPAddress(),
            'UsuarioEstado'     => $this->items['id'],
            'FechaEstado'       => $fechaActual,
            'MotivoEstado'      => $motivo,
            'Estado'            => $estado
        );
        
        $lastId = $this->areas->update([$idArea], $data);

		if($lastId !== FALSE){
			$message = sprintf($this->mensajeError->msg503, $idArea);
			echo $this->alertas->alertaExito($message);
			echo $this->alertas->refrescar(1);
		}
    }
    
    public function listarDirecciones()
    {
        if(isset($this->items['id'])){

			/* ACCIONES */
			$listadoDirecciones = $this->direccionModelo->listadoDireccion();
			
			$data = array (
				'session'               => 'on',
				'titulo'                => 'Listado de Direcciones | Verisure',
				'breadcrumb'            => $this->items['modulo'],
				'breadcrumb_alias'      => 'Direcciones',
				'listadoDirecciones'    => $listadoDirecciones
			);
			$data = array_merge($data, $this->items);
			return view('administracion/listar_direcciones',$data);			
		} else {
			return view('errors/500');
		}
    }
    
	public function registrarDireccion()
	{
        $fechaActual = date("Y-m-d H:i:s");
        
        /*
        * ----------------------------
        * DATOS DE AJAX PRE PROCESADOS
        * ----------------------------
        */
        
        $direccion = $this->request->getPost('txtDireccion_i') ;
        $sunat = $this->request->getPost('txtCodigoSunat_i');
        $distrito = $this->request->getPost('txtDistritoDireccion_i');
        
        /*
        * ----------------------------
        * VALIDACIÓN DE REQUERIMIENTOS
        * ----------------------------
        */
        
        $error = '';
        $error .= $this->complementos->validaCampo($direccion, 'required|trim|minlength[1]|maxlength[255]', 'Dirección');
        $error .= $this->complementos->validaCampo($sunat, 'required|trim|numeric', 'Código Sunat');
        $error .= $this->complementos->validaCampo($distrito, 'required|trim|numeric', 'Distrito');

        if ($error != '') {
            $message = sprintf($this->mensajeError->msg201, $error);
            echo $this->alertas->alertaError($message);
            EXIT;
        }
        
        /*
        * ----------------------------
        * VALIDACIÓN DE EXISTENCIA
        * ----------------------------
        */

        $msgExistencia = '';
        
		$direccionExistente = $this->direccionModelo->where('Direccion', $direccion)->findAll();
        
        if(count($direccionExistente)>0)
        {
            $msgExistencia .= "<li>La dirección ya fue registrada. [CAMPO] <small>DIRECCIÓN</small></li>";
        }

        if($msgExistencia != '')
        {
            $message = sprintf($this->mensajeError->msg201, $msgExistencia);
            echo $this->alertas->alertaPeligro($message);
            EXIT;
        }


        $data = array (
            'Direccion'             => $direccion,
            'id_distrito'           => $distrito,
            'fechahorareg'          => $fechaActual,
            'fija'                  => 1,
            'eliminacion_logica'    => 1,    
            'codigoSunat'           => $sunat
        );

        $lastId = $this->direccionModelo->insert($data, TRUE);

		if($lastId !== FALSE){
			$message = sprintf($this->mensajeError->msg502, $direccion);
			echo $this->alertas->alertaExito($message);
			echo $this->alertas->refrescar(1);
		}
	}
	
	public function obtenerDatosDireccion()
	{
		/*
        * ----------------------------
        * DATOS DE AJAX PRE PROCESADOS
        * ----------------------------
		*/

		$msjError = TRUE;
		$idDireccion = $this->request->getPost('idDireccion');

		/*
        * ----------------------------
        * VALIDACIÓN DE REQUERIMIENTOS
        * ----------------------------
        */

		$error = '';
		$error .= $this->complementos->validaCampo($idDireccion, 'trim', 'ID');

		if ($error != '') {
			$message = sprintf($this->mensajeError->msg201, $error);
			echo $this->alertas->alertaError($message);
			exit;
		}

		/*
        * -------------
        * BASE DE DATOS
        * -------------
        */
		
		$datosDireccion = $this->direccionModelo->obtenerDireccion($idDireccion);

		if(!empty($datosDireccion)){
			$resultado = array(
				'iddireccion'   => (string) $idDireccion,
				'direccion'     => (string) $datosDireccion['Direccion'],
				'sunat'         => (string) $datosDireccion['codigoSunat'],
				'iddpto'        => (string) $datosDireccion['id_dep_inei'],
				'idprovincia'   => (string) $datosDireccion['id_prov_inei'],
				'iddistrito'    => (string) $datosDireccion['id_distrito']
			);
		} else {
			$resultado = array(
				'descripcion' => 0
			);
		}

		/*
         * ----------------------------
         * DATOS DE AJAX PRE PROCESADOS
         * ----------------------------
         */

		echo json_encode($resultado);		
	}
	
	public function actualizarDireccion($idDireccion)
	{
        $fechaActual = date("Y-m-d H:i:s");
        
        /*
        * ----------------------------
        * DATOS DE AJAX PRE PROCESADOS
        * ----------------------------
        */
        
        $direccion = $this->request->getPost('txtDireccion_i') ;
        $sunat = $this->request->getPost('txtCodigoSunat_i');
        $distrito = $this->request->getPost('txtDistritoDireccion_i');
        
        /*
        * ----------------------------
        * VALIDACIÓN DE REQUERIMIENTOS
        * ----------------------------
        */
        
        $error = '';
        $error .= $this->complementos->validaCampo($direccion, 'required|trim|minlength[1]|maxlength[255]', 'Dirección');
        $error .= $this->complementos->validaCampo($sunat, 'required|trim|numeric', 'Código Sunat');
        $error .= $this->complementos->validaCampo($distrito, 'required|trim|numeric', 'Distrito');

        if ($error != '') {
            $message = sprintf($this->mensajeError->msg201, $error);
            echo $this->alertas->alertaError($message);
            EXIT;
        }
        
        /*
        * -------------
        * BASE DE DATOS
        * -------------
        */            
		
        $data = array (
            'Direccion'             => $direccion,
            'id_distrito'           => $distrito,
            'fechahorareg'          => $fechaActual, 
            'codigoSunat'           => $sunat
        );

        $lastId = $this->direccionModelo->update([$idDireccion], $data);

        if($lastId !== FALSE){
            $message = sprintf($this->mensajeError->msg503, $direccion);
            echo $this->alertas->alertaExito($message);
			echo $this->alertas->refrescar(1);
        }		
	}
	
    public function modificarEstadoDireccion()
    {
        $fechaActual = date("Y-m-d H:i:s");

        /*
        * ----------------------------
        * DATOS DE AJAX PRE PROCESADOS
        * ----------------------------
        */

        $msjError = TRUE;
        $idDireccion = $this->request->getPost('idDireccion');
        $estado = $this->request->getPost('estado');

        /*
        * ----------------------------
        * VALIDACIÓN DE REQUERIMIENTOS
        * ----------------------------
        */

        $error = '';
        $error .= $this->complementos->validaCampo($idDireccion, 'required|trim|numeric', 'ID Direccion');
        $error .= $this->complementos->validaCampo($estado, 'required|trim|numeric|minlength[1]|maxlength[1]', 'Estado');

        if ($error != '') {
            $message = sprintf($this->mensajeError->msg201, $error);
            echo $this->alertas->alertaError($message);
            EXIT;
        }

        $data = array (
            'eliminacion_logica'    => $estado
        );
        
        $lastId = $this->direccionModelo->update([$idDireccion], $data);

		if($lastId !== FALSE){
			$message = sprintf($this->mensajeError->msg503, $idDireccion);
			echo $this->alertas->alertaExito($message);
			echo $this->alertas->refrescar(1);
		}
    }
    
    public function listarUbicacionesAcceso($idDireccion)
    {
        if(isset($this->items['id'])){
            
			$listaUbicaciones = $this->ubicacionAcceso->where("IDDireccion", $idDireccion)->findAll();
            
			$data = array (
				'session'           => 'on',
				'titulo'            => 'Ubicaciones de Acceso | Verisure',
				'breadcrumb'        => $this->items['modulo'],
				'breadcrumb_alias'  => 'Ubicaciones de Acceso',
                'previo'            => 'listar-direcciones',
				'listaUbicaciones'  => $listaUbicaciones,
                'idDireccion'       => $idDireccion
			);
			$data = array_merge($data, $this->items);
			return view('administracion/listar_ubicaciones_acceso',$data);			
		} else {
			return view('errors/500');	
		}
    }
    
    public function registrarUbicacionAcceso($idDireccion)
    {
        $fechaActual = date("Y-m-d H:i:s");
        
        /*
        * ----------------------------
        * DATOS DE AJAX PRE PROCESADOS
        * ----------------------------
        */
        
        $msjError = TRUE;
        $ubicacion = ucwords(mb_strtolower($this->request->getPost('txtUbicacionAcceso')));
        
        /*
        * ----------------------------
        * VALIDACIÓN DE REQUERIMIENTOS
        * ----------------------------
        */
        
        $error = '';
        $error .= $this->complementos->validaCampo($ubicacion, 'required|trim|minlength[1]|maxlength[200]', 'Ubicación');

        if ($error != '') {
            $message = sprintf($this->mensajeError->msg201, $error);
            echo $this->alertas->alertaError($message);
            EXIT;
        }
        
        /*
        * ----------------------------
        * VALIDACIÓN DE EXISTENCIA
        * ----------------------------
        */

        $msgExistencia = '';
        
        $whereUbicacion = array("IDDireccion" => $idDireccion, 'Ubicacion', $ubicacion);
		$ubicacionExistente = $this->ubicacionAcceso->where($whereUbicacion)->findAll();
        
        if(count($ubicacionExistente)>0)
        {
            $msgExistencia .= "<li>La ubicación ya fue registrada. [CAMPO] <small>UBICACIÓN</small></li>";
        }

        if($msgExistencia != '')
        {
            $message = sprintf($this->mensajeError->msg201, $msgExistencia);
            echo $this->alertas->alertaPeligro($message);
            EXIT;
        }


        $data = array (
            'IDDireccion'   => $idDireccion, 
            'Ubicacion'     => $ubicacion,
            'UsuarioCrea'   => $this->items['id'],
            'FechaCrea'     => $fechaActual,
            'Estado'        => 'A'
        );

        $lastId = $this->ubicacionAcceso->insert($data, TRUE);

		if($lastId !== FALSE){
			$message = sprintf($this->mensajeError->msg502, $ubicacion);
			echo $this->alertas->alertaExito($message);
			echo $this->alertas->refrescar(1);
		}
    }

    
    public function listadoCargos()
    {
        //$elemento = $this->request->getPost('term');
        
        //$listado = $this->cargos->listadoCargosSelect($elemento['term']);

        $listado = $this->cargos->findAll();
        
        $elementos=array();

		if (!empty($listado)) {
            foreach ($listado as $items) {
                $elementos[] = array(
					'id' => (string) $items['IDCargo'],
                    'text' => (string) $items['Cargo']
                );
            }

            $resultado = array(
				'listadoCargos' => json_encode($elementos)
			);
        } else {
			$resultado = array(
				'descripcion' => 0,
			);
		}

        /*
         * ----------------------------
         * DATOS DE AJAX PRE PROCESADOS
         * ----------------------------
         */
        echo json_encode($resultado, JSON_NUMERIC_CHECK);		

    }

    public function listarCargos()
    {
        if(isset($this->items['id'])){

			/* ACCIONES */
			$listadoCargos = $this->cargos->findAll();
			
			$data = array (
				'session'               => 'on',
				'titulo'                => 'Listado de Cargos | Verisure',
				'breadcrumb'            => $this->items['modulo'],
				'breadcrumb_alias'      => 'Cargos',
				'listadoCargos'         => $listadoCargos
			);
			$data = array_merge($data, $this->items);
			return view('administracion/listar_cargos',$data);			
		} else {
			return view('errors/500');
		}
    }
    
	public function registrarCargo()
	{
        $fechaActual = date("Y-m-d H:i:s");
        
        /*
        * ----------------------------
        * DATOS DE AJAX PRE PROCESADOS
        * ----------------------------
        */
        
        $cargo = $this->request->getPost('txtCargo_i') ;
        $categoria = $this->request->getPost('txtCategoria_i');
        
        /*
        * ----------------------------
        * VALIDACIÓN DE REQUERIMIENTOS
        * ----------------------------
        */
        
        $error = '';
        $error .= $this->complementos->validaCampo($cargo, 'required|trim|minlength[1]|maxlength[100]', 'Cargo');
        // $error .= $this->complementos->validaCampo($categoria, 'required|trim|minlength[1]|maxlength[4]', 'Categoría');

        if ($error != '') {
            $message = sprintf($this->mensajeError->msg201, $error);
            echo $this->alertas->alertaError($message);
            EXIT;
        }
        
        /*
        * ----------------------------
        * VALIDACIÓN DE EXISTENCIA
        * ----------------------------
        */

        $msgExistencia = '';
        
		$cargoExistente = $this->cargos->where('Cargo', $cargo)->findAll();
        
        if(count($cargoExistente)>0)
        {
            $msgExistencia .= "<li>El cargo ya fue registrado. [CAMPO] <small>CARGO</small></li>";
        }

        if($msgExistencia != '')
        {
            $message = sprintf($this->mensajeError->msg201, $msgExistencia);
            echo $this->alertas->alertaPeligro($message);
            EXIT;
        }


        $data = array (
            'Cargo'         => $cargo,
            'Categoria'     => $categoria,
            'UsuarioCrea'   => $this->items['id'],
            'FechaCrea'     => $fechaActual,    
            'Localhost'     => $this->request->getIPAddress(),
            'UsuarioEstado' => $this->items['id'],
            'FechaEstado'   => $fechaActual,    
            'MotivoEstado'  => "Creación de cargo.",    
            'Estado'        => 'A'
        );

        $lastId = $this->cargos->insert($data, TRUE);

		if($lastId !== FALSE){
			$message = sprintf($this->mensajeError->msg502, $cargo);
			echo $this->alertas->alertaExito($message);
			echo $this->alertas->refrescar(1);
		}
	}
	
	public function obtenerDatosCargo()
	{
		/*
        * ----------------------------
        * DATOS DE AJAX PRE PROCESADOS
        * ----------------------------
		*/

		$msjError = TRUE;
		$idCargo = $this->request->getPost('idCargo');

		/*
        * ----------------------------
        * VALIDACIÓN DE REQUERIMIENTOS
        * ----------------------------
        */

		$error = '';
		$error .= $this->complementos->validaCampo($idCargo, 'trim', 'ID');

		if ($error != '') {
			$message = sprintf($this->mensajeError->msg201, $error);
			echo $this->alertas->alertaError($message);
			exit;
		}

		/*
        * -------------
        * BASE DE DATOS
        * -------------
        */
		
		$datosCargo = $this->cargos->where('IDCargo', $idCargo)->first();

		if(!empty($datosCargo)){
			$resultado = array(
				'idcargo'   => (string) $idCargo,
				'cargo'     => (string) $datosCargo['Cargo'],
				'categoria' => (string) $datosCargo['Categoria']
			);
		} else {
			$resultado = array(
				'descripcion' => 0
			);
		}

		/*
         * ----------------------------
         * DATOS DE AJAX PRE PROCESADOS
         * ----------------------------
         */

		echo json_encode($resultado);		
	}
	
	public function actualizarCargo($idCargo)
	{
        $fechaActual = date("Y-m-d H:i:s");
        
        /*
        * ----------------------------
        * DATOS DE AJAX PRE PROCESADOS
        * ----------------------------
        */
        
        $cargo = $this->request->getPost('txtCargo_i') ;
        $categoria = $this->request->getPost('txtCategoria_i');
        
        /*
        * ----------------------------
        * VALIDACIÓN DE REQUERIMIENTOS
        * ----------------------------
        */
        
        $error = '';
        $error .= $this->complementos->validaCampo($cargo, 'required|trim|minlength[1]|maxlength[100]', 'Cargo');
        $error .= $this->complementos->validaCampo($categoria, 'required|trim|minlength[1]|maxlength[4]', 'Categoría');

        if ($error != '') {
            $message = sprintf($this->mensajeError->msg201, $error);
            echo $this->alertas->alertaError($message);
            EXIT;
        }
        
        /*
        * -------------
        * BASE DE DATOS
        * -------------
        */            
		
        $data = array (
            'Cargo'             => $cargo,
            'Categoria'         => $categoria,
            'UsuarioModifica'   => $this->items['id'],
            'FechaModifica'     => $fechaActual,    
            'Localhost'         => $this->request->getIPAddress()
        );

        $lastId = $this->cargos->update([$idCargo], $data);

        if($lastId !== FALSE){
            $message = sprintf($this->mensajeError->msg503, $cargo);
            echo $this->alertas->alertaExito($message);
			echo $this->alertas->refrescar(1);
        }		
	}
	
    public function modificarEstadoCargo()
    {
        $fechaActual = date("Y-m-d H:i:s");

        /*
        * ----------------------------
        * DATOS DE AJAX PRE PROCESADOS
        * ----------------------------
        */

        $msjError = TRUE;
        $idCargo = $this->request->getPost('idCargo');
        $estado = $this->request->getPost('estado');
        $motivo = $this->request->getPost('motivo');

        /*
        * ----------------------------
        * VALIDACIÓN DE REQUERIMIENTOS
        * ----------------------------
        */

        $error = '';
        $error .= $this->complementos->validaCampo($idCargo, 'required|trim|numeric', 'ID Cargo');
        $error .= $this->complementos->validaCampo($estado, 'required|trim|alpha|minlength[1]|maxlength[1]', 'Estado');
        $error .= $this->complementos->validaCampo($motivo, 'required|trim', 'Motivo');

        if ($error != '') {
            $message = sprintf($this->mensajeError->msg201, $error);
            echo $this->alertas->alertaError($message);
            EXIT;
        }

        $data = array (
            'UsuarioModifica'   => $this->items['id'],
            'FechaModifica'     => $fechaActual,    
            'Localhost'         => $this->request->getIPAddress(),
            'UsuarioEstado'     => $this->items['id'],
            'FechaEstado'       => $fechaActual,
            'MotivoEstado'      => $motivo,
            'Estado'            => $estado
        );
        
        $lastId = $this->cargos->update([$idCargo], $data);

		if($lastId !== FALSE){
			$message = sprintf($this->mensajeError->msg503, $idCargo);
			echo $this->alertas->alertaExito($message);
			echo $this->alertas->refrescar(1);
		}
    }
    
    public function listarDireccionesEstructura()
    {
        if(isset($this->items['id'])){

			/* ACCIONES */
			$listadoDirecciones = $this->estructuraDirecciones->findAll();
			
			$data = array (
				'session'               => 'on',
				'titulo'                => 'Listado de Direcciones | Verisure',
				'breadcrumb'            => $this->items['modulo'],
				'breadcrumb_alias'      => 'Direcciones',
				'listadoDirecciones'    => $listadoDirecciones
			);
			$data = array_merge($data, $this->items);
			return view('administracion/listar_estructura_direcciones',$data);			
		} else {
			return view('errors/500');
		}
    }
    
	public function registrarDireccionEstructura()
	{
        $fechaActual = date("Y-m-d H:i:s");
        
        /*
        * ----------------------------
        * DATOS DE AJAX PRE PROCESADOS
        * ----------------------------
        */
        
        $direccion = ucwords(mb_strtolower($this->request->getPost('txtDireccionEstructura_i'))) ;
        $observacion = $this->request->getPost('txtObservacionDireccionEstructura_i');
        
        /*
        * ----------------------------
        * VALIDACIÓN DE REQUERIMIENTOS
        * ----------------------------
        */
        
        $error = '';
        $error .= $this->complementos->validaCampo($direccion, 'required|trim|minlength[1]|maxlength[100]', 'Dirección');

        if ($error != '') {
            $message = sprintf($this->mensajeError->msg201, $error);
            echo $this->alertas->alertaError($message);
            EXIT;
        }
        
        /*
        * ----------------------------
        * VALIDACIÓN DE EXISTENCIA
        * ----------------------------
        */

        $msgExistencia = '';
        
		$direccionExistente = $this->estructuraDirecciones->where('Direccion', $direccion)->findAll();
        
        if(count($direccionExistente)>0)
        {
            $msgExistencia .= "<li>La dirección ya fue registrada. [CAMPO] <small>DIRECCIÓN</small></li>";
        }

        if($msgExistencia != '')
        {
            $message = sprintf($this->mensajeError->msg201, $msgExistencia);
            echo $this->alertas->alertaPeligro($message);
            EXIT;
        }


        /*
        * ----------------------------
        * BASE DE DATOS
        * ----------------------------
        */

        $data = array (
            'Direccion'     => $direccion,
            'Observacion'   => $observacion,
            'UsuarioCrea'   => $this->items['id'],
            'FechaCrea'     => $fechaActual,    
            'Localhost'     => $this->request->getIPAddress(),
            'UsuarioEstado' => $this->items['id'],
            'FechaEstado'   => $fechaActual,    
            'MotivoEstado'  => "Creación de dirección.",    
            'Estado'        => 'A'
        );

        $lastId = $this->estructuraDirecciones->insert($data, TRUE);

		if($lastId !== FALSE){
			$message = sprintf($this->mensajeError->msg502, $direccion);
			echo $this->alertas->alertaExito($message);
			echo $this->alertas->refrescar(1);
		}
	}
	
	public function obtenerDatosDireccionEstructura()
	{
		/*
        * ----------------------------
        * DATOS DE AJAX PRE PROCESADOS
        * ----------------------------
		*/

		$msjError = TRUE;
		$idDireccion = $this->request->getPost('idDireccion');

		/*
        * ----------------------------
        * VALIDACIÓN DE REQUERIMIENTOS
        * ----------------------------
        */

		$error = '';
		$error .= $this->complementos->validaCampo($idDireccion, 'trim', 'ID');

		if ($error != '') {
			$message = sprintf($this->mensajeError->msg201, $error);
			echo $this->alertas->alertaError($message);
			exit;
		}

		/*
        * -------------
        * BASE DE DATOS
        * -------------
        */
		
		$datosDireccion = $this->estructuraDirecciones->where('IDDireccion', $idDireccion)->first();

		if(!empty($datosDireccion)){
			$resultado = array(
				'iddireccion'   => (string) $idDireccion,
				'direccion'     => (string) $datosDireccion['Direccion'],
				'observacion'   => (string) $datosDireccion['Observacion']
			);
		} else {
			$resultado = array(
				'descripcion' => 0
			);
		}

		/*
         * ----------------------------
         * DATOS DE AJAX PRE PROCESADOS
         * ----------------------------
         */

		echo json_encode($resultado);		
	}
	
	public function modificarDireccionEstructura($idDireccion)
	{
        $fechaActual = date("Y-m-d H:i:s");
        
        /*
        * ----------------------------
        * DATOS DE AJAX PRE PROCESADOS
        * ----------------------------
        */
        
        $direccion = ucwords(mb_strtolower($this->request->getPost('txtDireccionEstructura_i'))) ;
        $observacion = $this->request->getPost('txtObservacionDireccionEstructura_i');
        
        /*
        * ----------------------------
        * VALIDACIÓN DE REQUERIMIENTOS
        * ----------------------------
        */
        
        $error = '';
        $error .= $this->complementos->validaCampo($direccion, 'required|trim|minlength[1]|maxlength[100]', 'Dirección');

        if ($error != '') {
            $message = sprintf($this->mensajeError->msg201, $error);
            echo $this->alertas->alertaError($message);
            EXIT;
        }
        
        /*
        * -------------
        * BASE DE DATOS
        * -------------
        */            
		
        $data = array (
            'Direccion'         => $direccion,
            'Observacion'       => $observacion,
            'UsuarioModifica'   => $this->items['id'],
            'FechaModifica'     => $fechaActual,    
            'Localhost'         => $this->request->getIPAddress()
        );

        $lastId = $this->estructuraDirecciones->update([$idDireccion], $data);

        if($lastId !== FALSE){
            $message = sprintf($this->mensajeError->msg503, $direccion);
            echo $this->alertas->alertaExito($message);
			echo $this->alertas->refrescar(1);
        }		
	}
	
    public function modificarEstadoDireccionEstructura()
    {
        $fechaActual = date("Y-m-d H:i:s");

        /*
        * ----------------------------
        * DATOS DE AJAX PRE PROCESADOS
        * ----------------------------
        */

        $msjError = TRUE;
        $idDireccion = $this->request->getPost('idDireccion');
        $estado = $this->request->getPost('estado');
        $motivo = $this->request->getPost('motivo');

        /*
        * ----------------------------
        * VALIDACIÓN DE REQUERIMIENTOS
        * ----------------------------
        */

        $error = '';
        $error .= $this->complementos->validaCampo($idDireccion, 'required|trim|numeric', 'ID Cargo');
        $error .= $this->complementos->validaCampo($estado, 'required|trim|alpha|minlength[1]|maxlength[1]', 'Estado');
        $error .= $this->complementos->validaCampo($motivo, 'required|trim', 'Motivo');

        if ($error != '') {
            $message = sprintf($this->mensajeError->msg201, $error);
            echo $this->alertas->alertaError($message);
            EXIT;
        }

        $data = array (
            'UsuarioModifica'   => $this->items['id'],
            'FechaModifica'     => $fechaActual,    
            'Localhost'         => $this->request->getIPAddress(),
            'UsuarioEstado'     => $this->items['id'],
            'FechaEstado'       => $fechaActual,
            'MotivoEstado'      => $motivo,
            'Estado'            => $estado
        );
        
        $lastId = $this->estructuraDirecciones->update([$idDireccion], $data);

		if($lastId !== FALSE){
			$message = sprintf($this->mensajeError->msg503, $idDireccion);
			echo $this->alertas->alertaExito($message);
			echo $this->alertas->refrescar(1);
		}
    }
        
	public function listadoDireccionesEstructura(){
    	/*
        * ----------------------
        * OBTIENE Y VALIDA DATOS
        * ----------------------
        */
        $listadoDirecciones = $this->estructuraDirecciones->findAll();
        
        $direcciones=array();

		if (!empty($listadoDirecciones)) {
            foreach ($listadoDirecciones as $items) {
                $direcciones[] = array(
					'id' => (string) $items['IDDireccion'],
         	        'text' => (string) $items['Direccion'],
                );
            }

            $resultado = array(
				'listadoDirecciones' => json_encode($direcciones)
			);
        } else {
			$resultado = array(
				'descripcion' => 0,
			);
		}

        /*
         * ----------------------------
         * DATOS DE AJAX PRE PROCESADOS
         * ----------------------------
         */
        echo json_encode($resultado, JSON_NUMERIC_CHECK);		
	}

    public function listarAreasEstructura($idDireccion)
    {
        if(isset($this->items['id'])){

			/* ACCIONES */
            $direccion = $this->estructuraDirecciones->where('IDDireccion', $idDireccion)->first();
			$listaAreas = $this->estructuraAreas->where('IDDireccion', $idDireccion)->findAll();
			
			$data = array (
				'session'               => 'on',
				'titulo'                => 'Áreas | Verisure',
				'breadcrumb'            => $this->items['modulo'],
				'breadcrumb_alias'      => 'Áreas - '.$direccion['Direccion'],
                'previo'                => 'listar-direcciones-estructura',
                'direccion'             => $direccion,
				'listaAreas'            => $listaAreas
			);
			$data = array_merge($data, $this->items);
			return view('administracion/listar_estructura_areas',$data);			
		} else {
			return view('errors/500');
		}
    }
    
    public function registrarAreaEstructura()
    {
        $fechaActual = date("Y-m-d H:i:s");
        
        /*
        * ----------------------------
        * DATOS DE AJAX PRE PROCESADOS
        * ----------------------------
        */
        
        $msjError       = TRUE;
        $direccion      = $this->request->getPost('txtIDDireccionEstructura_i');
        $area           = ucwords(mb_strtolower($this->request->getPost('txtAreaEstructura_i')));
        $observacion    = $this->request->getPost('txtObservacionAreaEstructura_i');
        
        /*
        * ----------------------------
        * VALIDACIÓN DE REQUERIMIENTOS
        * ----------------------------
        */
        
        $error = '';
        $error .= $this->complementos->validaCampo($direccion, 'required|trim|number', 'ID Dirección');
        $error .= $this->complementos->validaCampo($area, 'required|trim|minlength[1]|maxlength[100]', 'Área');

        if ($error != '') {
            $message = sprintf($this->mensajeError->msg201, $error);
            echo $this->alertas->alertaError($message);
            EXIT;
        }
        
        /*
        * ----------------------------
        * VALIDACIÓN DE EXISTENCIA
        * ----------------------------
        */

        $msgExistencia = '';
        
        $whereArea = array(
            'IDDireccion'  => $direccion,
            'Area'   => $area
        );
        
		$areaExistente = $this->estructuraAreas->where($whereArea)->findAll();
        
        if(count($areaExistente)>0)
        {
            $msgExistencia .= "<li>El área fue registrado. [CAMPO] <small>ÁREA</small></li>";
        }

        if($msgExistencia != '')
        {
            $message = sprintf($this->mensajeError->msg201, $msgExistencia);
            echo $this->alertas->alertaPeligro($message);
            EXIT;
        }

        /*
        * ----------------------------
        * BASE DE DATOS
        * ----------------------------
        */

        $data = array (
            'IDDireccion'       => $direccion,
            'Area'              => $area,
            'Observacion'       => $observacion,
            'UsuarioCrea'       => $this->items['id'],
            'FechaCrea'         => $fechaActual,    
            'Localhost'         => $this->request->getIPAddress(),
            'Estado'            => 'A',
            'UsuarioEstado'     => $this->items['id'],
            'FechaEstado'       => $fechaActual,    
            'MotivoEstado'      => 'Área creada.'
        );

        $lastId = $this->estructuraAreas->insert($data, TRUE);

		if($lastId !== FALSE){
			$message = sprintf($this->mensajeError->msg502, $area);
			echo $this->alertas->alertaExito($message);
			echo $this->alertas->refrescar(1);
		}
    }

	public function obtenerDatosAreaEstructura() {

		/*
        * ----------------------------
        * DATOS DE AJAX PRE PROCESADOS
        * ----------------------------
		*/

		$msjError = TRUE;
		$idArea = $this->request->getPost('idArea');

		/*
        * ----------------------------
        * VALIDACIÓN DE REQUERIMIENTOS
        * ----------------------------
        */

		$error = '';
		$error .= $this->complementos->validaCampo($idArea, 'trim', 'ID');

		if ($error != '') {
			$message = sprintf($this->mensajeError->msg201, $error);
			echo $this->alertas->alertaError($message);
			exit;
		}

		/*
        * -------------
        * BASE DE DATOS
        * -------------
        */
		
		$datosArea = $this->estructuraAreas->where('IDArea', $idArea)->first();

		if(!empty($datosArea)){
			$resultado = array(
				'idarea'        => (string) $idArea,
				'area'          => (string) $datosArea['Area'],
				'observacion'   => (string) $datosArea['Observacion']
			);
		} else {
			$resultado = array(
				'descripcion' => 0
			);
		}

		/*
         * ----------------------------
         * DATOS DE AJAX PRE PROCESADOS
         * ----------------------------
         */

		echo json_encode($resultado);		
	}
	
    public function modificarAreaEstructura($idArea){

        $fechaActual = date("Y-m-d H:i:s");

        /*
        * ----------------------------
        * DATOS DE AJAX PRE PROCESADOS
        * ----------------------------
        */

        $msjError       = TRUE;
        $area           = ucwords(mb_strtolower($this->request->getPost('txtAreaEstructura_i')));
        $observacion    = $this->request->getPost('txtObservacionAreaEstructura_i');
        
        /*
        * ----------------------------
        * VALIDACIÓN DE REQUERIMIENTOS
        * ----------------------------
        */
        
        $error = '';
        $error .= $this->complementos->validaCampo($area, 'required|trim|minlength[1]|maxlength[50]', 'Área');

        if ($error != '') {
            $message = sprintf($this->mensajeError->msg201, $error);
            echo $this->alertas->alertaError($message);
            EXIT;
        }

        /*
        * ----------------------------
        * BASE DE DATOS
        * ----------------------------
        */
        
        $data = array (
            'Area'              => $area,
            'Observacion'       => $observacion,
            'UsuarioModifica'   => $this->items['id'],
            'FechaModifica'     => $fechaActual,    
            'Localhost'         => $this->request->getIPAddress()
        );

        $lastId = $this->estructuraAreas->update([$idArea], $data);

		if($lastId !== FALSE){
			$message = sprintf($this->mensajeError->msg503, $idArea);
			echo $this->alertas->alertaExito($message);
			echo $this->alertas->refrescar(1);
		}
    }
    
    public function modificarEstadoAreaEstructura()
    {
        $fechaActual = date("Y-m-d H:i:s");

        /*
        * ----------------------------
        * DATOS DE AJAX PRE PROCESADOS
        * ----------------------------
        */

        $msjError = TRUE;
        $idArea = $this->request->getPost('idArea');
        $estado = $this->request->getPost('estado');
        $motivo = $this->request->getPost('motivo');

        /*
        * ----------------------------
        * VALIDACIÓN DE REQUERIMIENTOS
        * ----------------------------
        */

        $error = '';
        $error .= $this->complementos->validaCampo($idArea, 'required|trim|numeric', 'ID Area');
        $error .= $this->complementos->validaCampo($estado, 'required|trim|alpha|minlength[1]|maxlength[1]', 'Estado');
        $error .= $this->complementos->validaCampo($motivo, 'required|trim', 'Motivo');

        if ($error != '') {
            $message = sprintf($this->mensajeError->msg201, $error);
            echo $this->alertas->alertaError($message);
            EXIT;
        }

        $data = array (
            'UsuarioModifica'   => $this->items['id'],
            'FechaModifica'     => $fechaActual,    
            'Localhost'         => $this->request->getIPAddress(),
            'Estado'            => $estado,
            'UsuarioEstado'     => $this->items['id'],
            'FechaEstado'       => $fechaActual,    
            'MotivoEstado'      => $motivo
        );
        
        $lastId = $this->estructuraAreas->update([$idArea], $data);

		if($lastId !== FALSE){
			$message = sprintf($this->mensajeError->msg503, $idArea);
			echo $this->alertas->alertaExito($message);
			echo $this->alertas->refrescar(1);
		}
    }
    
        
	public function listadoAreasEstructura(){
    	/*
        * ----------------------
        * OBTIENE Y VALIDA DATOS
        * ----------------------
        */
        
		$idDireccion = $this->request->getPost('idDireccion');

        if($idDireccion == 'TOTAL')
        {
            $listadoAreas = $this->estructuraAreas->findAll();
        }
        else
        {
            $whereAreas = array('IDDireccion' => $idDireccion);
            $listadoAreas = $this->estructuraAreas->where($whereAreas)->findAll();
        }
        
        $areas=array();

		if (!empty($listadoAreas)) {
            foreach ($listadoAreas as $items) {
                $areas[] = array(
					'id' => (string) $items['IDArea'],
         	        'text' => (string) $items['Area'],
                );
            }

            $resultado = array(
				'listadoAreas' => json_encode($areas)
			);
        } else {
			$resultado = array(
				'descripcion' => 0,
			);
		}

        /*
         * ----------------------------
         * DATOS DE AJAX PRE PROCESADOS
         * ----------------------------
         */
        echo json_encode($resultado, JSON_NUMERIC_CHECK);		
	}

    public function listarSubareasEstructura($idArea)
    {
        if(isset($this->items['id'])){

			/* ACCIONES */
            $area = $this->estructuraAreas->obtenerArea($idArea);
			$listaSubareas = $this->estructuraSubareas->where('IDArea', $idArea)->findAll();
			
			$data = array (
				'session'               => 'on',
				'titulo'                => 'Subáreas | Verisure',
				'breadcrumb'            => $this->items['modulo'],
				'breadcrumb_alias'      => 'Subáreas - '.$area['Area'].' - '.$area['Direccion'],
                'previo'                => 'listar-areas-estructura/'.$area['IDDireccion'],
                'area'                  => $area,
				'listaSubareas'         => $listaSubareas
			);
			$data = array_merge($data, $this->items);
			return view('administracion/listar_estructura_subareas',$data);			
		} else {
			return view('errors/500');
		}
    }
    
    public function registrarSubareaEstructura()
    {
        $fechaActual = date("Y-m-d H:i:s");
        
        /*
        * ----------------------------
        * DATOS DE AJAX PRE PROCESADOS
        * ----------------------------
        */
        
        $msjError       = TRUE;
        $area           = $this->request->getPost('txtIDAreaEstructura_i');
        $subarea        = ucwords(mb_strtolower($this->request->getPost('txtSubareaEstructura_i')));
        $correo    = $this->request->getPost('txtCorreoSubareaEstructura_i');
        $observacion    = $this->request->getPost('txtObservacionSubareaEstructura_i');
        
        /*
        * ----------------------------
        * VALIDACIÓN DE REQUERIMIENTOS
        * ----------------------------
        */
        
        $error = '';
        $error .= $this->complementos->validaCampo($area, 'required|trim|number', 'ID Área');
        $error .= $this->complementos->validaCampo($subarea, 'required|trim|minlength[1]|maxlength[100]', 'Subárea');

        if ($error != '') {
            $message = sprintf($this->mensajeError->msg201, $error);
            echo $this->alertas->alertaError($message);
            EXIT;
        }
        
        /*
        * ----------------------------
        * VALIDACIÓN DE EXISTENCIA
        * ----------------------------
        */

        $msgExistencia = '';
        
        $whereSubarea = array(
            'IDArea'    => $area,
            'Subarea'   => $subarea
        );
        
		$subareaExistente = $this->estructuraSubareas->where($whereSubarea)->findAll();
        
        if(count($subareaExistente)>0)
        {
            $msgExistencia .= "<li>La subárea fue registrada. [CAMPO] <small>SUBÁREA</small></li>";
        }

        if($msgExistencia != '')
        {
            $message = sprintf($this->mensajeError->msg201, $msgExistencia);
            echo $this->alertas->alertaPeligro($message);
            EXIT;
        }

        /*
        * ----------------------------
        * BASE DE DATOS
        * ----------------------------
        */

        $data = array (
            'IDArea'            => $area,
            'Subarea'           => $subarea,
            'Correo'            => $correo,
            'Observacion'       => $observacion,
            'UsuarioCrea'       => $this->items['id'],
            'FechaCrea'         => $fechaActual,    
            'Localhost'         => $this->request->getIPAddress(),
            'Estado'            => 'A',
            'UsuarioEstado'     => $this->items['id'],
            'FechaEstado'       => $fechaActual,    
            'MotivoEstado'      => 'Subárea creada.'
        );

        $lastId = $this->estructuraSubareas->insert($data, TRUE);

		if($lastId !== FALSE){
			$message = sprintf($this->mensajeError->msg502, $subarea);
			echo $this->alertas->alertaExito($message);
			echo $this->alertas->refrescar(1);
		}
    }

	public function obtenerDatosSubareaEstructura() {

		/*
        * ----------------------------
        * DATOS DE AJAX PRE PROCESADOS
        * ----------------------------
		*/

		$msjError = TRUE;
		$idSubarea = $this->request->getPost('idSubarea');

		/*
        * ----------------------------
        * VALIDACIÓN DE REQUERIMIENTOS
        * ----------------------------
        */

		$error = '';
		$error .= $this->complementos->validaCampo($idSubarea, 'trim', 'ID');

		if ($error != '') {
			$message = sprintf($this->mensajeError->msg201, $error);
			echo $this->alertas->alertaError($message);
			exit;
		}

		/*
        * -------------
        * BASE DE DATOS
        * -------------
        */
		
		$datosSubarea = $this->estructuraSubareas->where('IDSubarea', $idSubarea)->first();

		if(!empty($datosSubarea)){
			$resultado = array(
				'idsubarea'     => (string) $idSubarea,
				'subarea'       => (string) $datosSubarea['Subarea'],
				'correo'        => (string) $datosSubarea['Correo'],
				'observacion'   => (string) $datosSubarea['Observacion']
			);
		} else {
			$resultado = array(
				'descripcion' => 0
			);
		}

		/*
         * ----------------------------
         * DATOS DE AJAX PRE PROCESADOS
         * ----------------------------
         */

		echo json_encode($resultado);		
	}
	
    public function modificarSubareaEstructura($idSubarea){

        $fechaActual = date("Y-m-d H:i:s");

        /*
        * ----------------------------
        * DATOS DE AJAX PRE PROCESADOS
        * ----------------------------
        */

        $msjError       = TRUE;
        $subarea        = ucwords(mb_strtolower($this->request->getPost('txtSubareaEstructura_i')));
        $observacion    = $this->request->getPost('txtObservacionSubareaEstructura_i');
        $correo         = $this->request->getPost('txtCorreoSubareaEstructura_i');
        
        /*
        * ----------------------------
        * VALIDACIÓN DE REQUERIMIENTOS
        * ----------------------------
        */
        
        $error = '';
        $error .= $this->complementos->validaCampo($subarea, 'required|trim|minlength[1]|maxlength[100]', 'Subárea');

        if ($error != '') {
            $message = sprintf($this->mensajeError->msg201, $error);
            echo $this->alertas->alertaError($message);
            EXIT;
        }

        /*
        * ----------------------------
        * BASE DE DATOS
        * ----------------------------
        */
        
        $data = array (
            'Subarea'           => $subarea,
            'Correo'            => $correo,
            'Observacion'       => $observacion,
            'UsuarioModifica'   => $this->items['id'],
            'FechaModifica'     => $fechaActual,    
            'Localhost'         => $this->request->getIPAddress()
        );

        $lastId = $this->estructuraSubareas->update([$idSubarea], $data);

		if($lastId !== FALSE){
			$message = sprintf($this->mensajeError->msg503, $idSubarea);
			echo $this->alertas->alertaExito($message);
			echo $this->alertas->refrescar(1);
		}
    }
    
    public function modificarEstadoSubareaEstructura()
    {
        $fechaActual = date("Y-m-d H:i:s");

        /*
        * ----------------------------
        * DATOS DE AJAX PRE PROCESADOS
        * ----------------------------
        */

        $msjError = TRUE;
        $idSubarea = $this->request->getPost('idSubarea');
        $estado = $this->request->getPost('estado');
        $motivo = $this->request->getPost('motivo');

        /*
        * ----------------------------
        * VALIDACIÓN DE REQUERIMIENTOS
        * ----------------------------
        */

        $error = '';
        $error .= $this->complementos->validaCampo($idSubarea, 'required|trim|numeric', 'ID Subárea');
        $error .= $this->complementos->validaCampo($estado, 'required|trim|alpha|minlength[1]|maxlength[1]', 'Estado');
        $error .= $this->complementos->validaCampo($motivo, 'required|trim', 'Motivo');

        if ($error != '') {
            $message = sprintf($this->mensajeError->msg201, $error);
            echo $this->alertas->alertaError($message);
            EXIT;
        }

        $data = array (
            'UsuarioModifica'   => $this->items['id'],
            'FechaModifica'     => $fechaActual,    
            'Localhost'         => $this->request->getIPAddress(),
            'Estado'            => $estado,
            'UsuarioEstado'     => $this->items['id'],
            'FechaEstado'       => $fechaActual,    
            'MotivoEstado'      => $motivo
        );
        
        $lastId = $this->estructuraSubareas->update([$idSubarea], $data);

		if($lastId !== FALSE){
			$message = sprintf($this->mensajeError->msg503, $idSubarea);
			echo $this->alertas->alertaExito($message);
			echo $this->alertas->refrescar(1);
		}
    }
    
    public function listadoSubareasEstructura()
    {
        /*
        * ----------------------
        * OBTIENE Y VALIDA DATOS
        * ----------------------
        */
        
		$idArea = $this->request->getPost('idArea');

        if($idArea == 'TOTAL')
        {
            $listadoSubareas = $this->estructuraSubareas->findAll();
        }
        else
        {
            $whereSubareas = array('IDArea' => $idArea);
            $listadoSubareas = $this->estructuraSubareas->where($whereSubareas)->findAll();
        }
        
        $subareas=array();

		if (!empty($listadoSubareas)) {
            foreach ($listadoSubareas as $items) {
                $subareas[] = array(
					'id' => (string) $items['IDSubarea'],
         	        'text' => (string) $items['Subarea'],
                );
            }

            $resultado = array(
				'listadoSubareas' => json_encode($subareas)
			);
        } else {
			$resultado = array(
				'descripcion' => 0,
			);
		}

        /*
         * ----------------------------
         * DATOS DE AJAX PRE PROCESADOS
         * ----------------------------
         */
        echo json_encode($resultado, JSON_NUMERIC_CHECK);
    }
    
    public function listadoUsuarioSubareasEstructura()
    {
        /*
        * ----------------------
        * OBTIENE Y VALIDA DATOS
        * ----------------------
        */
        
		$idSubArea = $this->request->getPost('idSubArea');

        if($idSubArea == 'TOTAL')
        {
            $listadoUsuariosSubareas = $this->usuarioModelo->findAll();
        }
        else
        {
            $whereSubareas = array('IDSubarea' => $idSubArea);
            $listadoUsuariosSubareas = $this->usuarioModelo->where($whereSubareas)->findAll();
        }
        
        $usuariosSubareas=array();

		if (!empty($listadoUsuariosSubareas)) {
            foreach ($listadoUsuariosSubareas as $items) {
                $usuariosSubareas[] = array(
					'id' => (string) $items['IdUsuario'],
         	        'text' => (string) $items['Matricula'].' | '.$items['Nombres'].' '.$items['Apellidos'],
                );
            }

            $resultado = array(
				'listadoUsuariosSubareas' => json_encode($usuariosSubareas)
			);
        } else {
			$resultado = array(
				'descripcion' => 0,
			);
		}

        /*
         * ----------------------------
         * DATOS DE AJAX PRE PROCESADOS
         * ----------------------------
         */
        echo json_encode($resultado, JSON_NUMERIC_CHECK);
    }
    
    public function listarDelegacionessEstructura()
    {
        if(isset($this->items['id'])){

			/* ACCIONES */
			$listadoDelegaciones = $this->estructuraDelegaciones->findAll();
			
			$data = array (
				'session'               => 'on',
				'titulo'                => 'Delegaciones | Verisure',
				'breadcrumb'            => $this->items['modulo'],
				'breadcrumb_alias'      => 'Listado de delegaciones',
				'listadoDelegaciones'   => $listadoDelegaciones
			);
			$data = array_merge($data, $this->items);
			return view('administracion/listar_estructura_delegaciones',$data);			
		} else {
			return view('errors/500');
		}
    }
    
	public function registrarDelegacionEstructura()
	{
        $fechaActual = date("Y-m-d H:i:s");
        
        /*
        * ----------------------------
        * DATOS DE AJAX PRE PROCESADOS
        * ----------------------------
        */
        
        $delegacion = ucwords(mb_strtolower($this->request->getPost('txtDelegacionEstructura_i'))) ;
        $cc = $this->request->getPost('txtCCDelegacionEstructura_i');
        $ln = $this->request->getPost('txtLNDelegacionEstructura_i');
        $observacion = $this->request->getPost('txtObservacionDelegacionEstructura_i');
        
        /*
        * ----------------------------
        * VALIDACIÓN DE REQUERIMIENTOS
        * ----------------------------
        */
        
        $error = '';
        $error .= $this->complementos->validaCampo($delegacion, 'required|trim|minlength[1]|maxlength[100]', 'Delegación');
        $error .= $this->complementos->validaCampo($cc, 'required|trim|numeric', 'CC');
        $error .= $this->complementos->validaCampo($ln, 'required|trim|numeric', 'LN');

        if ($error != '') {
            $message = sprintf($this->mensajeError->msg201, $error);
            echo $this->alertas->alertaError($message);
            EXIT;
        }
        
        /*
        * ----------------------------
        * VALIDACIÓN DE EXISTENCIA
        * ----------------------------
        */

        $msgExistencia = '';
        
        $whereDelegaciones = array(
            'CC'        => $cc,
            'LN'        => $ln
        );
        
		$combinacionExistente = $this->estructuraDelegaciones->where($whereDelegaciones)->findAll();
		
        if(count($combinacionExistente)>0)
        {
            $msgExistencia .= "<li>La combinación de CC y LN ya fue agregada. [CAMPO] <small>CC-LN</small></li>";
        }

        if($msgExistencia != '')
        {
            $message = sprintf($this->mensajeError->msg201, $msgExistencia);
            echo $this->alertas->alertaPeligro($message);
            EXIT;
        }


        $data = array (
            'Delegacion'    => $delegacion,
            'CC'            => $cc,
            'LN'            => $ln,
            'Observacion'   => $observacion,
            'UsuarioCrea'   => $this->items['id'],
            'FechaCrea'     => $fechaActual,    
            'Localhost'     => $this->request->getIPAddress(),
            'UsuarioEstado' => $this->items['id'],
            'FechaEstado'   => $fechaActual,    
            'MotivoEstado'  => "Creación de delegación.",    
            'Estado'        => 'A'
        );

        $lastId = $this->estructuraDelegaciones->insert($data, TRUE);

		if($lastId !== FALSE){
			$message = sprintf($this->mensajeError->msg502, $delegacion);
			echo $this->alertas->alertaExito($message);
			echo $this->alertas->refrescar(1);
		}
	}
	
	public function obtenerDatosDelegacionEstructura()
	{
		/*
        * ----------------------------
        * DATOS DE AJAX PRE PROCESADOS
        * ----------------------------
		*/

		$msjError = TRUE;
		$idDelegacion = $this->request->getPost('idDelegacion');

		/*
        * ----------------------------
        * VALIDACIÓN DE REQUERIMIENTOS
        * ----------------------------
        */

		$error = '';
		$error .= $this->complementos->validaCampo($idDelegacion, 'trim', 'ID');

		if ($error != '') {
			$message = sprintf($this->mensajeError->msg201, $error);
			echo $this->alertas->alertaError($message);
			exit;
		}

		/*
        * -------------
        * BASE DE DATOS
        * -------------
        */
		
		$datosDelegacion = $this->estructuraDelegaciones->where('IDDelegacion', $idDelegacion)->first();

		if(!empty($datosDelegacion)){
			$resultado = array(
				'iddelegacion'  => (string) $idDelegacion,
				'delegacion'    => (string) $datosDelegacion['Delegacion'],
				'cc'            => (string) $datosDelegacion['CC'],
				'ln'            => (string) $datosDelegacion['LN'],
				'observacion'   => (string) $datosDelegacion['Observacion']
			);
		} else {
			$resultado = array(
				'descripcion' => 0
			);
		}

		/*
         * ----------------------------
         * DATOS DE AJAX PRE PROCESADOS
         * ----------------------------
         */

		echo json_encode($resultado);		
	}
	
	public function modificarDelegacionEstructura($idDelegacion)
	{
        $fechaActual = date("Y-m-d H:i:s");
        
        /*
        * ----------------------------
        * DATOS DE AJAX PRE PROCESADOS
        * ----------------------------
        */
        
        $delegacion = ucwords(mb_strtolower($this->request->getPost('txtDelegacionEstructura_i'))) ;
        $cc = $this->request->getPost('txtCCDelegacionEstructura_i');
        $ln = $this->request->getPost('txtLNDelegacionEstructura_i');
        $observacion = $this->request->getPost('txtObservacionDelegacionEstructura_i');
        
        /*
        * ----------------------------
        * VALIDACIÓN DE REQUERIMIENTOS
        * ----------------------------
        */
        
        $error = '';
        $error .= $this->complementos->validaCampo($delegacion, 'required|trim|minlength[1]|maxlength[100]', 'Delegación');
        $error .= $this->complementos->validaCampo($cc, 'required|trim|numeric', 'CC');
        $error .= $this->complementos->validaCampo($ln, 'required|trim|numeric', 'LN');

        if ($error != '') {
            $message = sprintf($this->mensajeError->msg201, $error);
            echo $this->alertas->alertaError($message);
            EXIT;
        }
        
        /*
        * -------------
        * BASE DE DATOS
        * -------------
        */            
		
        $data = array (
            'Delegacion'        => $delegacion,
            'CC'                => $cc,
            'LN'                => $ln,
            'Observacion'       => $observacion,
            'UsuarioModifica'   => $this->items['id'],
            'FechaModifica'     => $fechaActual,    
            'Localhost'         => $this->request->getIPAddress()
        );

        $lastId = $this->estructuraDelegaciones->update([$idDelegacion], $data);

        if($lastId !== FALSE){
            $message = sprintf($this->mensajeError->msg503, $delegacion);
            echo $this->alertas->alertaExito($message);
			echo $this->alertas->refrescar(1);
        }		
	}
	
    public function modificarEstadoDelegacionEstructura()
    {
        $fechaActual = date("Y-m-d H:i:s");

        /*
        * ----------------------------
        * DATOS DE AJAX PRE PROCESADOS
        * ----------------------------
        */

        $msjError = TRUE;
        $idDelegacion = $this->request->getPost('idDelegacion');
        $estado = $this->request->getPost('estado');
        $motivo = $this->request->getPost('motivo');

        /*
        * ----------------------------
        * VALIDACIÓN DE REQUERIMIENTOS
        * ----------------------------
        */

        $error = '';
        $error .= $this->complementos->validaCampo($idDelegacion, 'required|trim|numeric', 'ID Delegación');
        $error .= $this->complementos->validaCampo($estado, 'required|trim|alpha|minlength[1]|maxlength[1]', 'Estado');
        $error .= $this->complementos->validaCampo($motivo, 'required|trim', 'Motivo');

        if ($error != '') {
            $message = sprintf($this->mensajeError->msg201, $error);
            echo $this->alertas->alertaError($message);
            EXIT;
        }

        $data = array (
            'UsuarioModifica'   => $this->items['id'],
            'FechaModifica'     => $fechaActual,    
            'Localhost'         => $this->request->getIPAddress(),
            'UsuarioEstado'     => $this->items['id'],
            'FechaEstado'       => $fechaActual,
            'MotivoEstado'      => $motivo,
            'Estado'            => $estado
        );
        
        $lastId = $this->estructuraDelegaciones->update([$idDelegacion], $data);

		if($lastId !== FALSE){
			$message = sprintf($this->mensajeError->msg503, $idDelegacion);
			echo $this->alertas->alertaExito($message);
			echo $this->alertas->refrescar(1);
		}
    }
    
	public function listadoDelegacionesEstructura(){
    	/*
        * ----------------------
        * OBTIENE Y VALIDA DATOS
        * ----------------------
        */
        $listadoDelegaciones = $this->estructuraDelegaciones->findAll();
        
        $delegaciones=array();

		if (!empty($listadoDelegaciones)) {
            foreach ($listadoDelegaciones as $items) {
                $delegaciones[] = array(
					'id' => (string) $items['IDDelegacion'],
         	        'text' => (string) $items['Delegacion'].' | '.$items['CC'].' | '.$items['LN'],
                );
            }

            $resultado = array(
				'listadoDelegaciones' => json_encode($delegaciones)
			);
        } else {
			$resultado = array(
				'descripcion' => 0,
			);
		}

        /*
         * ----------------------------
         * DATOS DE AJAX PRE PROCESADOS
         * ----------------------------
         */
        echo json_encode($resultado, JSON_NUMERIC_CHECK);		
	}
	
	public function busquedaDelegacionesEstructura()
	{
	    $elemento = $this->request->getPost('term');
	    
	    $array = array("variable" => $elemento['term']);
        
        $listado = $this->estructuraDelegaciones->listadoDelegacionesBusqueda($array);
        
        $elementos=array();

        foreach ($listado as $items) {
            $elementos[] = array(
                'id' => (string) $items['IDDelegacion'],
                'text' => (string) $items['CC'].'-'.$items['LN'].'|'.$items['Delegacion']
            );
        }

        echo json_encode($elementos);
	}

    public function listarDelegacionesSubareaEstructura($idSubarea)
    {
        if(isset($this->items['id'])){
			/* BUSCAR ACCIONES */
			$subarea = $this->estructuraSubareas->obtenerSubarea($idSubarea);
			$listaDelegaciones = $this->estructuraSubareaDelegaciones->listarDelegacionesXSubarea($idSubarea);
            
			$data = array (
				'session'           => 'on',
				'titulo'            => 'Delegaciones - Subárea | Verisure',
				'breadcrumb'        => $this->items['modulo'],
				'breadcrumb_alias'  => 'Delegaciones asociadas - '.$subarea['Subarea'],
                'previo'            => 'listar-subareas-estructura/'.$subarea['IDArea'],
				'listaDelegaciones' => $listaDelegaciones,
                'idSubareaDel'         => $idSubarea
			);
			$data = array_merge($data, $this->items);
			return view('administracion/listar_estructura_subarea_delegaciones',$data);			
		} else {
			return view('errors/500');	
		}
    }
    
    public function asociarDelegacionSubareaEstructura()
    {
        $fechaActual = date("Y-m-d H:i:s");
        
        /*
        * ----------------------------
        * DATOS DE AJAX PRE PROCESADOS
        * ----------------------------
        */
        
        $msjError = TRUE;
        $idDelegacion = $this->request->getPost('idDelegacion');
        $idSubarea = $this->request->getPost('idSubarea');
        
        /*
        * ----------------------------
        * VALIDACIÓN DE REQUERIMIENTOS
        * ----------------------------
        */
        
        $error = '';
        $error .= $this->complementos->validaCampo($idDelegacion, 'required|trim|number', 'ID Delegación');
        $error .= $this->complementos->validaCampo($idSubarea, 'required|trim|number', 'ID Subárea');

        if ($error != '') {
            $message = sprintf($this->mensajeError->msg201, $error);
            echo $this->alertas->alertaError($message);
            EXIT;
        }
        
        /*
        * ----------------------------
        * VALIDACIÓN DE EXISTENCIA
        * ----------------------------
        */

        $msgExistencia = '';
        
        $whereCC = array(
            'IDSubarea'     => $idSubarea,
            'IDDelegacion'  => $idDelegacion
        );
        
		$combinacionExistente = $this->estructuraSubareaDelegaciones->where($whereCC)->findAll();
		
        if(count($combinacionExistente)>0)
        {
            $msgExistencia .= "<li>La delegación ya fue agregada a la subárea. [CAMPO] <small>DELEGACIÓN</small></li>";
        }

        if($msgExistencia != '')
        {
            $message = sprintf($this->mensajeError->msg201, $msgExistencia);
            echo $this->alertas->alertaPeligro($message);
            EXIT;
        }

        /*
        * ----------------------------
        * BASE DE DATOS
        * ----------------------------
        */

        $data = array (
            'IDSubarea'         => $idSubarea, 
            'IDDelegacion'      => $idDelegacion,
            'UsuarioRegistra'   => $this->items['id'],
            'FechaRegistra'     => $fechaActual,
            'Localhost'         => $this->request->getIPAddress()
        );

        $lastId = $this->estructuraSubareaDelegaciones->insert($data, TRUE);

		if($lastId !== FALSE){
			$message = sprintf($this->mensajeError->msg502, $idDelegacion);
			echo $this->alertas->alertaExito($message);
			echo $this->alertas->refrescar(1);
		}
    }
    
    public function eliminarDelegacionSubareaEstructura()
    {
        /*
        * ----------------------------
        * DATOS DE AJAX PRE PROCESADOS
        * ----------------------------
        */
        
        $msjError = TRUE;
        $idSubarea = $this->request->getPost('idSubarea') ;
        $idDelegacion = $this->request->getPost('idDelegacion');
        
        /*
        * ----------------------------
        * VALIDACIÓN DE REQUERIMIENTOS
        * ----------------------------
        */
        
        $error = '';
        $error .= $this->complementos->validaCampo($idSubarea, 'required|trim|number', 'ID Subárea');
        $error .= $this->complementos->validaCampo($idDelegacion, 'required|trim|number', 'ID Delegación');
        
        if ($error != '') {
            $message = sprintf($this->mensajeError->msg201, $error);
            echo $this->alertas->alertaError($message);
            EXIT;
        }
        
        /*
        * ----------------------------
        * BASE DE DATOS
        * ----------------------------
        */

        $whereDelete = array (
            'IDSubarea'     => $idSubarea, 
            'IDDelegacion'  => $idDelegacion
        );

        $lastId = $this->estructuraSubareaDelegaciones->where($whereDelete)->delete();

		if($lastId !== FALSE){
			$message = sprintf($this->mensajeError->msg504, $idDelegacion);
			echo $this->alertas->alertaExito($message);
			echo $this->alertas->refrescar(1);
		}
    }

    public function listadoDelegacionSubareaEstructura()
    {
        /*
        * ----------------------
        * OBTIENE Y VALIDA DATOS
        * ----------------------
        */
        
		$idSubarea = $this->request->getPost('idSubarea');

        if($idSubarea == 'TOTAL')
        {
            $listadoDelegaciones = $this->estructuraDelegaciones->findAll();
        }
        else
        {
            $listadoDelegaciones = $this->estructuraSubareaDelegaciones->listarDelegacionesXSubarea($idSubarea);
        }
        
        $delegaciones=array();

		if (!empty($listadoDelegaciones)) {
            foreach ($listadoDelegaciones as $items) {
                $delegaciones[] = array(
					'id' => (string) $items['IDDelegacion'],
         	        'text' => (string) $items['CC'].'-'.$items['LN'].' | '.$items['Delegacion'],
                );
            }

            $resultado = array(
				'listadoDelegaciones' => json_encode($delegaciones)
			);
        } else {
			$resultado = array(
				'descripcion' => 0,
			);
		}
		
        /*
         * ----------------------------
         * DATOS DE AJAX PRE PROCESADOS
         * ----------------------------
         */
        echo json_encode($resultado, JSON_NUMERIC_CHECK);
    }
    
}