<?php

namespace SpoutExample;

use SpoutExample\Iterator\BatchFetchIterator;
use SpoutExample\Iterator\SingleFetchIterator;
use SpoutExample\ReportWriter\PHPExcelWriter;
use SpoutExample\ReportWriter\SpoutWriter;
use SpoutExample\ReportWriter\WriterType;

/**
 * Class ReportCreatorReferidos
 * In this example, we want to generate a report listing all the products available
 * as well as some associated data (quantity available, quantity sold, ...).
 *
 * @package SpoutExample
 */
class ReportCreatorReferidos
{
    /** Fetching methods */
    const FETCH_ROWS_ONE_BY_ONE = 1;
    const FETCH_ROWS_IN_BATCH = 2;
    const FETCH_ROWS_ALL_AT_ONCE = 3;

    /** @var \PDO PDO instance */
    private $pdo;

    /** @var ReportWriter\WriterType Type of writer to use */
    private $writerType = WriterType::SPOUT;

    /** @var ReportWriter\AbstractWriter Writer used to create a XLSX report */
    private $reportWriter;

    /** @var int The fetching method to use, used for benchmarks */
    private $fetchingMethod = self::FETCH_ROWS_IN_BATCH;

    /** @var int Number of rows to fetch for each batch (used only when FETCH_ROWS_IN_BATCH is enabled) */
    private $batchSize;

    /**
     * @param DBConf $dbConf
     */
    public function __construct(DBConf $dbConf)
    {
        $this->pdo = new \PDO($dbConf->getDSN(), $dbConf->getUsername(), $dbConf->getPassword());
    }

    /**
     * @return self
     */
    public function setFetchRowsOneByOne()
    {
        $this->fetchingMethod = self::FETCH_ROWS_ONE_BY_ONE;
        return $this;
    }

    /**
     * @param int $batchSize Number of rows to fetch for each batch
     * @return self
     */
    public function setFetchRowsInBatch($batchSize)
    {
        $this->fetchingMethod = self::FETCH_ROWS_IN_BATCH;
        $this->batchSize = $batchSize;
        return $this;
    }

    /**
     * @return self
     */
    public function setFetchAllRowsAtOnce()
    {
        $this->fetchingMethod = self::FETCH_ROWS_ALL_AT_ONCE;
        return $this;
    }

    /**
     * @return string Name of the fetching method used
     */
    public function getFetchMethodName()
    {
        switch ($this->fetchingMethod) {
            case self::FETCH_ROWS_ONE_BY_ONE: return 'Fetch mode: one by one';
            case self::FETCH_ROWS_ALL_AT_ONCE: return 'Fetch mode: all at once';
            case self::FETCH_ROWS_IN_BATCH:
            default:
                return 'Fetch mode: batch';
        }
    }

    /**
     * @see \SpoutExample\ReportWriter\WriterType
     *
     * @param string $writerType
     * @return ReportCreator
     */
    public function setWriterType($writerType)
    {
        $this->writerType = $writerType;
        return $this;
    }

    /**
     * @param $outputPath
     * @return ReportWriter\AbstractWriter
     */
    private function createReportWriter($outputPath)
    {
        switch ($this->writerType) {
            case WriterType::PHP_EXCEL: return new PHPExcelWriter($outputPath);
            case WriterType::SPOUT:
            default:
                return new SpoutWriter($outputPath);
        }
    }

    /**
     * Fetches the data from the DB and spits it out in a XLSX file.
     *
     * @param string $outputPath Path where the report will be written to
     * @return void
     */
    public function fetchDataAndCreateReport($outputPath)
    {
        $this->reportWriter = $this->createReportWriter($outputPath);
        $this->writeReportHeader();

        // Make sure to only select the fields we are interested in
        $query = "SELECT 
v.IdVenta,
v.IdUsuario,
v.IdCliente,
v.Matricula,
v.Comercial,
v.CC,
v.JefeVenta,
v.CategoriaVenta,
v.TipoVenta,
v.Segmento,
v.NProspecto,
v.NContrato,
v.PrecioKit,
v.TotalExtra,
v.IngresoKIT,
v.IngresoExtras,
v.IngresoTotal,
v.EstadoVenta,
v.ExtraAutorizado,
v.MesesAutorizado,
v.N590,
v.FormaPago,
v.PAI,
v.ResumenVenta,
v.FechaHoraREG,
v.EstadoVisita,
v.FechaVisita,
v.HoraVisita,
v.MesVisita,
v.FVencimientoVisita,
v.GestorMKT,
v.OrigenVisita,
v.ComentarioVisita,
v.GestorOrigen,
c.IdGeografico,
c.NombApellido,
c.Telefonouno,
c.Telefonodos,
c.Correo,
c.Direccion,
g.Distrito,
g.Provincia,
v.CCOriginal,
v.EstadoTelecierre,
v.TCGestor,
v.TCMotivoNoVenta,
v.TCTipoVenta,
v.Checkin,
v.LeadID,
concat(a.FechaInst, ' ', a.HoraInst) FechaHoraInst,
d.TipoDocumento,
d.Documento,
v.Accion,
v.PerfilScoring,
v.IdConsultaScoring,
s.Distrital as DistritalScoring,
s.Provincial as ProvincialScoring,
s.Departamental as DepartamentalScoring,
s.GestorOrigen as MatriculaScoring,
s.Score,
s.PredictorIngresos_IP,
s.CantidadDocumentosMorosos,
s.FechaHoraReg as FechaHoraConsulta,
v.IMP,
c.objGeofinder
FROM yifi58ge_dbVENTAS.tblventa v INNER JOIN yifi58ge_dbVENTAS.tblcliente c on c.IdCliente=v.IdCliente
INNER JOIN yifi58ge_dbVENTAS.tblagenda a ON a.IdVenta=v.IdVenta 
INNER JOIN yifi58ge_dbVENTAS.tblgeografico g ON g.IdGeografico=c.IdGeografico
LEFT JOIN yifi58ge_dbVENTAS.tbldocumento d ON c.IdDocVisita = d.IdDoc
LEFT JOIN yifi58ge_dbVENTAS.tbltmpscoring s ON s.IdConsultaScoring=v.IdConsultaScoring    
WHERE v.TipoVenta='RE'
and v.IdVenta >= 45451";

        switch ($this->fetchingMethod) {
            case self::FETCH_ROWS_ONE_BY_ONE:
                $this->fetchRowsOneByOneAndWriteThem($query);
                break;
            case self::FETCH_ROWS_IN_BATCH:
                $this->fetchRowsInBatchAndWriteThem($query);
                break;
            case self::FETCH_ROWS_ALL_AT_ONCE:
                $this->fetchAllRowsAtOnceAndWriteThem($query);
                break;
        }

        $this->reportWriter->close();
    }

    /**
     * @param string $query
     * @param int $maxFetchedRows
     * @return void
     */
    private function fetchRowsOneByOneAndWriteThem($query)
    {
        $dbRowIterator = new SingleFetchIterator($this->pdo, $query);

        foreach ($dbRowIterator as $dbRow) {
                if(!empty($dbRow['objGeofinder']) || !is_null($dbRow['objGeofinder'])){
                    $objGeofinder = json_decode($dbRow['objGeofinder'], true);
                    //$InfoCheckIn = json_decode($Venta['InfoCheckIn'], true);
                    $zonas = $objGeofinder['zonas'];
                    //ZONA
                    if(!empty($zonas[0]['valor'])){
                       $zona = $zonas[0]['valor']; 
                    }else{
                      $zona = '';  
                    }
                    //MICROZONA
                    if(!empty($zonas[1]['valor'])){
                       //$microzona = $zonas[1]['valor']; 
                       
                       $microzona = convertirMicrozona($zonas[1]['valor']);
                       
                    }else{
                      $microzona = '';  
                    }                    
                }else{
                    //ZONA
                    $zona = '';
                    //MICROZONA
                    $microzona='';
                }
            
            
            $reportRow = [$dbRow['IdVenta'],
$dbRow['IdUsuario'],
$dbRow['IdCliente'],
$dbRow['Matricula'],
$dbRow['Comercial'],
$dbRow['CC'],
$dbRow['JefeVenta'],
$dbRow['CategoriaVenta'],
$dbRow['TipoVenta'],
$dbRow['Segmento'],
$dbRow['NProspecto'],
$dbRow['NContrato'],
$dbRow['PrecioKit'],
$dbRow['TotalExtra'],
$dbRow['IngresoKIT'],
$dbRow['IngresoExtras'],
$dbRow['IngresoTotal'],
$dbRow['EstadoVenta'],
$dbRow['ExtraAutorizado'],
$dbRow['MesesAutorizado'],
$dbRow['N590'],
$dbRow['FormaPago'],
$dbRow['PAI'],
$dbRow['ResumenVenta'],
$dbRow['FechaHoraREG'],
$dbRow['EstadoVisita'],
$dbRow['FechaVisita'],
$dbRow['HoraVisita'],
$dbRow['MesVisita'],
$dbRow['FVencimientoVisita'],
$dbRow['GestorMKT'],
$dbRow['OrigenVisita'],
$dbRow['ComentarioVisita'],
$dbRow['GestorOrigen'],
$dbRow['IdGeografico'],
$dbRow['NombApellido'],
$dbRow['Telefonouno'],
$dbRow['Telefonodos'],
$dbRow['Correo'],
$dbRow['Direccion'],
$dbRow['Distrito'],
$dbRow['Provincia'],
$dbRow['CCOriginal'],
$dbRow['EstadoTelecierre'],
$dbRow['TCGestor'],
$dbRow['TCMotivoNoVenta'],
$dbRow['TCTipoVenta'],
$dbRow['Checkin'],
$dbRow['LeadID'],
$dbRow['FechaHoraInst'],
$dbRow['TipoDocumento'],
$dbRow['Documento'],
$dbRow['Accion'],
$dbRow['PerfilScoring'],
$dbRow['IdConsultaScoring'],
$dbRow['DistritalScoring'],
$dbRow['ProvincialScoring'],
$dbRow['DepartamentalScoring'],
$dbRow['MatriculaScoring'],
$dbRow['Score'],
$dbRow['PredictorIngresos_IP'],
$dbRow['CantidadDocumentosMorosos'],
$dbRow['FechaHoraConsulta'],
$zona,
$microzona,
$dbRow['IMP']
];
            $this->reportWriter->writeRow($reportRow);
        }
    }

    /**
     * @param string $query
     * @param int $maxFetchedRows
     * @return void
     */
    private function fetchRowsInBatchAndWriteThem($query)
    {
        $idFieldName = 'IdVenta';
        $batchFetchIterator = new BatchFetchIterator($this->pdo, $query, $idFieldName, $this->batchSize);

        foreach ($batchFetchIterator as $dbRows) {
            foreach ($dbRows as $dbRow) {
                $reportRow = [$dbRow['IdVenta'], $dbRow['IdUsuario'], $dbRow['IdCliente'], $dbRow['Matricula'], $dbRow['Comercial'], $dbRow['CC']];
                $this->reportWriter->writeRow($reportRow);
            }
        }
    }

    /**
     * @param string $query
     * @return void
     */
    private function fetchAllRowsAtOnceAndWriteThem($query)
    {
        $statement = $this->pdo->prepare($query);
        $statement->execute();

        $allDBRows = $statement->fetchAll(\PDO::FETCH_ASSOC);

        foreach ($allDBRows as $dbRow) {
            $reportRow =  [$dbRow['IdVenta'], $dbRow['IdUsuario'], $dbRow['IdCliente'], $dbRow['Matricula'], $dbRow['Comercial'], $dbRow['CC']];
            $this->reportWriter->writeRow($reportRow);
        }

        $statement->closeCursor();
    }

    /**
     * @return void
     */
    private function writeReportHeader()
    {
        // The header will be bold
        $headerRow = ['IdVenta',
'IdUsuario',
'IdCliente',
'Matricula',
'Comercial',
'CC',
'JefeVenta',
'CategoriaVenta',
'TipoVenta',
'Segmento',
'NProspecto',
'NContrato',
'PrecioKit',
'TotalExtra',
'IngresoKIT',
'IngresoExtras',
'IngresoTotal',
'EstadoVenta',
'ExtraAutorizado',
'MesesAutorizado',
'N590',
'FormaPago',
'PAI',
'ResumenVenta',
'FechaHoraREG',
'EstadoVisita',
'FechaVisita',
'HoraVisita',
'MesVisita',
'FVencimientoVisita',
'GestorMKT',
'OrigenVisita',
'ComentarioVisita',
'GestorOrigen',
'IdGeografico',
'NombApellido',
'Telefonouno',
'Telefonodos',
'Correo',
'Direccion',
'Distrito',
'Provincia',
'CCOriginal',
'EstadoTelecierre',
'TCGestor',
'TCMotivoNoVenta',
'TCTipoVenta',
'Checkin',
'LeadID',
'FechaHoraInst',
'TipoDocumento',
'Documento',
'Accion',
'PerfilScoring',
'IdConsultaScoring',
'DistritalScoring',
'ProvincialScoring',
'DepartamentalScoring',
'MatriculaScoring',
'Score',
'PredictorIngresos_IP',
'CantidadDocumentosMorosos',
'FechaHoraConsulta',
'Zona',
'Seccion',
'IMP'];
        $this->reportWriter->writeHeaderRow($headerRow);
    }
    //FUNCIONES DE DESCARGA DE ReferidosRP
    public function fetchDataAndCreateReport_ReferidosRP($outputPath){
        $this->reportWriter = $this->createReportWriter($outputPath);
        $this->writeReportHeader_ReferidosRP();
        // Make sure to only select the fields we are interested in
        $query="SELECT DISTINCT 
        v.IdProspectoRP,
        v.TipoVenta,
        v.FechaHoraREG as FechaCreacionProspectoRP,
        ven.IdVenta,
        ven.NContrato,
        v.EstadoVenta,
        v.referidoHabilitado,
        t.NroReferidos,
        a.FechaInst 
        FROM yifi58ge_dbVENTAS.tblprospectorp v 
        LEFT JOIN yifi58ge_dbREFERIDO.referidosRP rfe on v.IdProspectoRP=rfe.IdProspectoRP 
        LEFT JOIN yifi58ge_dbREFERIDO.totalreferidos t on t.IdProspectoRP=rfe.IdProspectoRP 
        INNER JOIN yifi58ge_dbVENTAS.tblventa ven on ven.IdProspectoRP=v.IdProspectoRP 
        INNER JOIN yifi58ge_dbVENTAS.tblagenda a on a.IdVenta=ven.IdVenta
        WHERE a.FechaInst>='2021-11-16' and (ven.NContrato is NOT null) and ven.NContrato <>''
        ";
        switch ($this->fetchingMethod) {
            case self::FETCH_ROWS_ONE_BY_ONE:
                $this->fetchRowsOneByOneAndWriteThem_ReferidosRP($query);
                break;
            case self::FETCH_ROWS_IN_BATCH:
                $this->fetchRowsInBatchAndWriteThem($query);
                break;
            case self::FETCH_ROWS_ALL_AT_ONCE:
                $this->fetchAllRowsAtOnceAndWriteThem($query);
                break;
        }

        $this->reportWriter->close();
    }
    private function writeReportHeader_ReferidosRP(){
        // The header will be bold
        $headerRow = [
        'IdProspectoRP',
        'TipoVenta',
        'FechaCreacionProspectoRP',
        'IdVenta',
        'NContrato',
        'EstadoVenta',
        'referidoHabilitado',
        'NroReferidos',
        'FechaInst' 
            ];
        $this->reportWriter->writeHeaderRow($headerRow);
    }
    private function fetchRowsOneByOneAndWriteThem_ReferidosRP($query){
        $dbRowIterator = new SingleFetchIterator($this->pdo, $query);
        foreach ($dbRowIterator as $dbRow) {
        $reportRow = [
            $dbRow['IdProspectoRP'],
            $dbRow['TipoVenta'],
            $dbRow['FechaCreacionProspectoRP'],
            $dbRow['IdVenta'],
            $dbRow['NContrato'],
            $dbRow['EstadoVenta'],
            $dbRow['referidoHabilitado'],
            $dbRow['NroReferidos'],
            $dbRow['FechaInst']
            ];
            $this->reportWriter->writeRow($reportRow);
        }
    }     

  //FIN ReferidosRP
  
      //FUNCIONES DE DESCARGA DE ReferidosRPPR
    public function fetchDataAndCreateReport_ReferidosRPPR($outputPath){
        $this->reportWriter = $this->createReportWriter($outputPath);
        $this->writeReportHeader_ReferidosRPPR();
        // Make sure to only select the fields we are interested in
        $query="SELECT Id, PrimerNombre, SegundoNombre, ApellidoPaterno, ApellidoMaterno, TipoDocumento, Documento, Telefono, Correo, FechaNacimiento, Direccion, Distrito, Provincia, Departamento, CodUbigeoReniec, FechaHoraREG, MatriculaREG, EstadoReferido FROM yifi58ge_dbREFERIDO.tblreferidorppr";
        switch ($this->fetchingMethod) {
            case self::FETCH_ROWS_ONE_BY_ONE:
                $this->fetchRowsOneByOneAndWriteThem_ReferidosRPPR($query);
                break;
            case self::FETCH_ROWS_IN_BATCH:
                $this->fetchRowsInBatchAndWriteThem($query);
                break;
            case self::FETCH_ROWS_ALL_AT_ONCE:
                $this->fetchAllRowsAtOnceAndWriteThem($query);
                break;
        }

        $this->reportWriter->close();
    }
    private function writeReportHeader_ReferidosRPPR(){
        // The header will be bold
        $headerRow = [
            'Id',
            'PrimerNombre',
            'SegundoNombre',
            'ApellidoPaterno',
            'ApellidoMaterno',
            'TipoDocumento',
            'Documento',
            'Telefono',
            'Correo',
            'FechaNacimiento',
            'Direccion',
            'Distrito',
            'Provincia',
            'Departamento',
            'CodUbigeoReniec',
            'FechaHoraREG',
            'MatriculaREG',
            'EstadoReferido'
            ];
        $this->reportWriter->writeHeaderRow($headerRow);
    }
    private function fetchRowsOneByOneAndWriteThem_ReferidosRPPR($query){
        $dbRowIterator = new SingleFetchIterator($this->pdo, $query);
        foreach ($dbRowIterator as $dbRow) {
        $reportRow = [
                $dbRow['Id'],
                $dbRow['PrimerNombre'],
                $dbRow['SegundoNombre'],
                $dbRow['ApellidoPaterno'],
                $dbRow['ApellidoMaterno'],
                $dbRow['TipoDocumento'],
                "'".$dbRow['Documento'],
                $dbRow['Telefono'],
                $dbRow['Correo'],
                $dbRow['FechaNacimiento'],
                $dbRow['Direccion'],
                $dbRow['Distrito'],
                $dbRow['Provincia'],
                $dbRow['Departamento'],
                $dbRow['CodUbigeoReniec'],
                $dbRow['FechaHoraREG'],
                $dbRow['MatriculaREG'],
                $dbRow['EstadoReferido']
            ];
            $this->reportWriter->writeRow($reportRow);
        }
    }     

  //FIN ReferidosRPPR
  
  
  //FUNCIONES DE DESCARGA DE REFERIDOSRE
    public function fetchDataAndCreateReport_ReferidosRE($outputPath){
        $this->reportWriter = $this->createReportWriter($outputPath);
        $this->writeReportHeader_ReferidosRE();
        // Make sure to only select the fields we are interested in
        $query="SELECT DISTINCT 
        v.IdProspectoRP,
        v.TipoVenta,
        v.FechaHoraREG as FechaCreacionProspectoRE,
        v.IdVenta,
        v.NContrato,
        v.EstadoVenta,
        v.referidoHabilitado,
        t.NroReferidos,
        a.FechaInst 
        FROM yifi58ge_dbVENTAS.tblventa v 
        LEFT JOIN yifi58ge_dbREFERIDO.referidosRE rfe on v.IdVenta=rfe.IdProspectoRP
        LEFT JOIN yifi58ge_dbREFERIDO.totalreferidos t on t.IdProspectoRP=rfe.IdProspectoRP
        INNER JOIN yifi58ge_dbVENTAS.tblagenda a on a.IdVenta=v.IdVenta
        WHERE a.FechaInst>='2021-11-16' and (v.NContrato is NOT null) and v.NContrato <>'' and v.CC<>512
        ";
        switch ($this->fetchingMethod) {
            case self::FETCH_ROWS_ONE_BY_ONE:
                $this->fetchRowsOneByOneAndWriteThem_ReferidosRE($query);
                break;
            case self::FETCH_ROWS_IN_BATCH:
                $this->fetchRowsInBatchAndWriteThem($query);
                break;
            case self::FETCH_ROWS_ALL_AT_ONCE:
                $this->fetchAllRowsAtOnceAndWriteThem($query);
                break;
        }

        $this->reportWriter->close();
    }
    private function writeReportHeader_ReferidosRE(){
        // The header will be bold
        $headerRow = [
        'IdProspectoRP',
        'TipoVenta',
        'FechaCreacionProspectoRE',
        'IdVenta',
        'NContrato',
        'EstadoVenta',
        'referidoHabilitado',
        'NroReferidos',
        'FechaInst' 
            ];
        $this->reportWriter->writeHeaderRow($headerRow);
    }
    private function fetchRowsOneByOneAndWriteThem_ReferidosRE($query){
        $dbRowIterator = new SingleFetchIterator($this->pdo, $query);
        foreach ($dbRowIterator as $dbRow) {
        $reportRow = [
            $dbRow['IdProspectoRP'],
            $dbRow['TipoVenta'],
            $dbRow['FechaCreacionProspectoRE'],
            $dbRow['IdVenta'],
            $dbRow['NContrato'],
            $dbRow['EstadoVenta'],
            $dbRow['referidoHabilitado'],
            $dbRow['NroReferidos'],
            $dbRow['FechaInst']
            ];
            $this->reportWriter->writeRow($reportRow);
        }
    }       

  //FIN REFERIDOS
  
    //FUNCIONES DE DESCARGA DE REFERIDOS
    public function fetchDataAndCreateReport_Referidos($outputPath){
        $this->reportWriter = $this->createReportWriter($outputPath);
        $this->writeReportHeader_Referidos();
        // Make sure to only select the fields we are interested in
        $query="SELECT refd.Id,ref.IdProspectoRP,refd.EstadoReferido,ref.TipoVenta,pro.Matricula as MatriculaVendedor,pro.Comercial as NombreVendedor
        ,pro.CC as Delegacion,v.IdVenta,v.NContrato,a.FechaInst,refd.PrimerNombre,refd.PrimerApellido,refd.Telefono,refd.Segmento,refd.FechaHoraREG,refd.TipoContacto,refd.GestorCall,refd.FechaPrimerContacto,refd.FechaUltimoContacto,refd.ComentarioCallRP,refd.Vueltas,refd.FechaRellamada,refd.HoraRellamada,promkt.IdProspectoRP as IdProspectoRPTMK,promkt.EstadoProspecto as EstadoProspectoRPTMK,promkt.ComentarioProspectoRP as ComentarioProspectoRPTMK,promkt.FechaVisitaRP as FechaVisitaRPTMK,promkt.HoraVisitaRP as HoraVisitaRPTMK,vmkt.IdVenta as IdVentaRPTMK,vmkt.NContrato as NContratoRPTMK,amkt.FechaInst as FechaInstRPTMK
         FROM yifi58ge_dbREFERIDO.tblreferido_detalle refd 
        INNER JOIN yifi58ge_dbREFERIDO.tblreferido ref on ref.Id=refd.Referido_ID
        INNER JOIN yifi58ge_dbVENTAS.tblprospectorp pro on pro.IdProspectoRP=ref.IdProspectoRP
        LEFT JOIN yifi58ge_dbVENTAS.tblventa v on v.IdProspectoRP=pro.IdProspectoRP
        LEFT JOIN yifi58ge_dbVENTAS.tblagenda a on a.IdVenta=v.IdVenta
        LEFT JOIN yifi58ge_dbVENTAS.tblprospectorp promkt on promkt.IdReferido=refd.Id
        LEFT JOIN yifi58ge_dbVENTAS.tblventa vmkt on vmkt.IdProspectoRP=promkt.IdProspectoRP
        LEFT JOIN yifi58ge_dbVENTAS.tblagenda amkt on amkt.IdVenta=vmkt.IdVenta
        WHERE ref.TipoVenta in ('RP','RPPR','RPPA','RPTMK')
        UNION
        SELECT refd.Id,ref.IdProspectoRP,refd.EstadoReferido,ref.TipoVenta,v.Matricula as MatriculaVendedor,v.Comercial as NombreVendedor
        ,v.CC as Delegacion,v.IdVenta,v.NContrato,a.FechaInst,refd.PrimerNombre,refd.PrimerApellido,refd.Telefono,refd.Segmento,refd.FechaHoraREG,refd.TipoContacto,refd.GestorCall,refd.FechaPrimerContacto,refd.FechaUltimoContacto,refd.ComentarioCallRP,refd.Vueltas,refd.FechaRellamada,refd.HoraRellamada,promkt.IdProspectoRP as IdProspectoRPTMK,promkt.EstadoProspecto as EstadoProspectoRPTMK,promkt.ComentarioProspectoRP as ComentarioProspectoRPTMK,promkt.FechaVisitaRP as FechaVisitaRPTMK,promkt.HoraVisitaRP as HoraVisitaRPTMK,vmkt.IdVenta as IdVentaRPTMK,vmkt.NContrato as NContratoRPTMK,amkt.FechaInst as FechaInstRPTMK
         FROM yifi58ge_dbREFERIDO.tblreferido_detalle refd 
        INNER JOIN yifi58ge_dbREFERIDO.tblreferido ref on ref.Id=refd.Referido_ID
        INNER JOIN yifi58ge_dbVENTAS.tblventa v on v.IdVenta=ref.IdProspectoRP
        INNER JOIN yifi58ge_dbVENTAS.tblagenda a on a.IdVenta=v.IdVenta
        LEFT JOIN yifi58ge_dbVENTAS.tblprospectorp promkt on promkt.IdReferido=refd.Id
        LEFT JOIN yifi58ge_dbVENTAS.tblventa vmkt on vmkt.IdProspectoRP=promkt.IdProspectoRP
        LEFT JOIN yifi58ge_dbVENTAS.tblagenda amkt on amkt.IdVenta=vmkt.IdVenta
        WHERE ref.TipoVenta in ('RE','RPMKT')
        ";
        switch ($this->fetchingMethod) {
            case self::FETCH_ROWS_ONE_BY_ONE:
                $this->fetchRowsOneByOneAndWriteThem_Referidos($query);
                break;
            case self::FETCH_ROWS_IN_BATCH:
                $this->fetchRowsInBatchAndWriteThem($query);
                break;
            case self::FETCH_ROWS_ALL_AT_ONCE:
                $this->fetchAllRowsAtOnceAndWriteThem($query);
                break;
        }

        $this->reportWriter->close();
    }
    private function writeReportHeader_Referidos(){
        // The header will be bold
        $headerRow = [
            'Id',
            'IdProspectoRP',
            'EstadoReferido',
            'TipoVenta',
            'MatriculaVendedor',
            'NombreVendedor',
            'Delegacion',
            'IdVenta',
            'NContrato',
            'FechaInst',
            'PrimerNombre',
            'PrimerApellido',
            'Telefono',
            'Segmento',
            'FechaHoraREG',
            'TipoContacto',
            'GestorCall',
            'FechaPrimerContacto',
            'FechaUltimoContacto',
            'ComentarioCallRP',
            'Vueltas',
            'FechaRellamada',
            'HoraRellamada',
            'IdProspectoRPTMK',
            'EstadoProspectoRPTMK',
            'ComentarioProspectoRPTMK',
            'FechaVisitaRPTMK',
            'HoraVisitaRPTMK',
            'IdVentaRPTMK',
            'NContratoRPTMK',
            'FechaInstRPTMK'
            ];
        $this->reportWriter->writeHeaderRow($headerRow);
    }
    private function fetchRowsOneByOneAndWriteThem_Referidos($query){
        $dbRowIterator = new SingleFetchIterator($this->pdo, $query);
        foreach ($dbRowIterator as $dbRow) {
        $reportRow = [
            $dbRow['Id'],
            $dbRow['IdProspectoRP'],
            $dbRow['EstadoReferido'],
            $dbRow['TipoVenta'],
            $dbRow['MatriculaVendedor'],
            $dbRow['NombreVendedor'],
            $dbRow['Delegacion'],
            $dbRow['IdVenta'],
            $dbRow['NContrato'],
            $dbRow['FechaInst'],
            $dbRow['PrimerNombre'],
            $dbRow['PrimerApellido'],
            $dbRow['Telefono'],
            $dbRow['Segmento'],
            $dbRow['FechaHoraREG'],
            $dbRow['TipoContacto'],
            $dbRow['GestorCall'],
            $dbRow['FechaPrimerContacto'],
            $dbRow['FechaUltimoContacto'],
            $dbRow['ComentarioCallRP'],
            $dbRow['Vueltas'],
            $dbRow['FechaRellamada'],
            $dbRow['HoraRellamada'],
            $dbRow['IdProspectoRPTMK'],
            $dbRow['EstadoProspectoRPTMK'],
            $dbRow['ComentarioProspectoRPTMK'],
            $dbRow['FechaVisitaRPTMK'],
            $dbRow['HoraVisitaRPTMK'],
            $dbRow['IdVentaRPTMK'],
            $dbRow['NContratoRPTMK'],
            $dbRow['FechaInstRPTMK']
            ];
            $this->reportWriter->writeRow($reportRow);
        }
    }       

  //FIN REFERIDOSRE

    private function convertirMicrozona($microzona){
        $nuevaMicrozona = '';
        
        if(strpos($microzona, '-')){
            
            $arrayMicrozona = explode("-", $microzona);
            
            $ubigeo = $arrayMicrozona[0];
            
            $correlativo = '';
            
            $arregloUbigeos = array('150132',
                                '150133',
                                '130101',
                                '130104',
                                '130105',
                                '130111',
                                '130103',
                                '130106',
                                '130107',
                                '130109',
                                '130102',
                                '200101',
                                '200104',
                                '200105',
                                '200107',
                                '200108',
                                '200109',
                                '200110',
                                '200111',
                                '200114',
                                '200115',
                                '040122',
                                '040117',
                                '040112',
                                '040129',
                                '040128',
                                '040104',
                                '040103',
                                '040126',
                                '040101',
                                '040111',
                                '040116',
                                '040107',
                                '040109',
                                '040110',
                                '040123',
                                '040124',
                                '040102',
                                '040120',
                                '040118',
                                '040108',
                                '040125',
                                '040121',
                                '040114',
                                '040113',
                                '040119',
                                '040105',
                                '040106');
                                
            //var_dump($arregloUbigeos);
            
            //exit();
                                
            if (in_array($ubigeo, $arregloUbigeos)) {
                $correlativo = $arrayMicrozona[1];
                
                $correlativo = substr($correlativo, 1, 4);
                
                $nuevaMicrozona = $ubigeo.$correlativo;
                
                
                
            } else{
                $correlativo = $arrayMicrozona[1];
                
                $correlativo = substr($correlativo, 2, 4);
                
                $nuevaMicrozona = $ubigeo.$correlativo;
                
            }
            
            return $nuevaMicrozona;
            
            //return true;    
        } else{
            //return false;
             return $microzona;
        }
    }

    
}
