<?php

namespace SpoutExample;

use SpoutExample\Iterator\BatchFetchIterator;
use SpoutExample\Iterator\SingleFetchIterator;
use SpoutExample\ReportWriter\PHPExcelWriter;
use SpoutExample\ReportWriter\SpoutWriter;
use SpoutExample\ReportWriter\WriterType;

/**
 * Class ReportCreatorVisitas
 * In this example, we want to generate a report listing all the products available
 * as well as some associated data (quantity available, quantity sold, ...).
 *
 * @package SpoutExample
 */
class ReportCreatorTelecierre
{
    /** Fetching methods */
    const FETCH_ROWS_ONE_BY_ONE = 1;
    const FETCH_ROWS_IN_BATCH = 2;
    const FETCH_ROWS_ALL_AT_ONCE = 3;

    /** @var \PDO PDO instance */
    private $pdo;

    /** @var ReportWriter\WriterType Type of writer to use */
    private $writerType = WriterType::SPOUT;

    /** @var ReportWriter\AbstractWriter Writer used to create a XLSX report */
    private $reportWriter;

    /** @var int The fetching method to use, used for benchmarks */
    private $fetchingMethod = self::FETCH_ROWS_IN_BATCH;

    /** @var int Number of rows to fetch for each batch (used only when FETCH_ROWS_IN_BATCH is enabled) */
    private $batchSize;

    /**
     * @param DBConf $dbConf
     */
    public function __construct(DBConf $dbConf)
    {
        $this->pdo = new \PDO($dbConf->getDSN(), $dbConf->getUsername(), $dbConf->getPassword());
    }

    /**
     * @return self
     */
    public function setFetchRowsOneByOne()
    {
        $this->fetchingMethod = self::FETCH_ROWS_ONE_BY_ONE;
        return $this;
    }

    /**
     * @param int $batchSize Number of rows to fetch for each batch
     * @return self
     */
    public function setFetchRowsInBatch($batchSize)
    {
        $this->fetchingMethod = self::FETCH_ROWS_IN_BATCH;
        $this->batchSize = $batchSize;
        return $this;
    }

    /**
     * @return self
     */
    public function setFetchAllRowsAtOnce()
    {
        $this->fetchingMethod = self::FETCH_ROWS_ALL_AT_ONCE;
        return $this;
    }

    /**
     * @return string Name of the fetching method used
     */
    public function getFetchMethodName()
    {
        switch ($this->fetchingMethod) {
            case self::FETCH_ROWS_ONE_BY_ONE: return 'Fetch mode: one by one';
            case self::FETCH_ROWS_ALL_AT_ONCE: return 'Fetch mode: all at once';
            case self::FETCH_ROWS_IN_BATCH:
            default:
                return 'Fetch mode: batch';
        }
    }

    /**
     * @see \SpoutExample\ReportWriter\WriterType
     *
     * @param string $writerType
     * @return ReportCreator
     */
    public function setWriterType($writerType)
    {
        $this->writerType = $writerType;
        return $this;
    }

    /**
     * @param $outputPath
     * @return ReportWriter\AbstractWriter
     */
    private function createReportWriter($outputPath)
    {
        switch ($this->writerType) {
            case WriterType::PHP_EXCEL: return new PHPExcelWriter($outputPath);
            case WriterType::SPOUT:
            default:
                return new SpoutWriter($outputPath);
        }
    }

    /**
     * Fetches the data from the DB and spits it out in a XLSX file.
     *
     * @param string $outputPath Path where the report will be written to
     * @return void
     */
    public function fetchDataAndCreateReport($outputPath)
    {
        $this->reportWriter = $this->createReportWriter($outputPath);
        $this->writeReportHeader();

        // Make sure to only select the fields we are interested in
        $query = "SELECT 
v.IdVenta,
v.IdUsuario,
v.IdCliente,
v.Matricula,
v.Comercial,
v.CC,
v.JefeVenta,
v.CategoriaVenta,
v.TipoVenta,
v.Segmento,
v.NProspecto,
v.NContrato,
v.PrecioKit,
v.TotalExtra,
v.IngresoKIT,
v.IngresoExtras,
v.IngresoTotal,
v.EstadoVenta,
v.ExtraAutorizado,
v.MesesAutorizado,
v.N590,
v.FormaPago,
v.PAI,
v.ResumenVenta,
v.FechaHoraREG,
v.EstadoVisita,
v.FechaVisita,
v.HoraVisita,
v.MesVisita,
v.FVencimientoVisita,
v.GestorMKT,
v.OrigenVisita,
v.ComentarioVisita,
v.GestorOrigen,
c.IdGeografico,
c.NombApellido,
c.Telefonouno,
c.Telefonodos,
c.Correo,
c.Direccion,
g.Distrito,
g.Provincia,
v.CCOriginal,
v.EstadoTelecierre,
v.TCGestor,
v.TCMotivoNoVenta,
v.TCTipoVenta,
v.Checkin,
v.LeadID,
concat(a.FechaInst, ' ', a.HoraInst) FechaHoraInst,
d.TipoDocumento,
d.Documento,
v.Accion,
v.PerfilScoring,
v.IdConsultaScoring,
s.Distrital as DistritalScoring,
s.Provincial as ProvincialScoring,
s.Departamental as DepartamentalScoring,
s.GestorOrigen as MatriculaScoring,
s.Score,
s.PredictorIngresos_IP,
s.CantidadDocumentosMorosos,
s.FechaHoraReg as FechaHoraConsulta,
v.IMP,
c.objGeofinder,
v.pagoAdelantado
FROM yifi58ge_dbVENTAS.tblventa v INNER JOIN yifi58ge_dbVENTAS.tblcliente c on c.IdCliente=v.IdCliente
INNER JOIN yifi58ge_dbVENTAS.tblagenda a ON a.IdVenta=v.IdVenta 
INNER JOIN yifi58ge_dbVENTAS.tblgeografico g ON g.IdGeografico=c.IdGeografico
LEFT JOIN yifi58ge_dbVENTAS.tbldocumento d ON c.IdDocVisita = d.IdDoc
LEFT JOIN yifi58ge_dbVENTAS.tbltmpscoring s ON s.IdConsultaScoring=v.IdConsultaScoring    
WHERE v.TipoVenta='RE'
and v.IdVenta >= 45451";

        switch ($this->fetchingMethod) {
            case self::FETCH_ROWS_ONE_BY_ONE:
                $this->fetchRowsOneByOneAndWriteThem($query);
                break;
            case self::FETCH_ROWS_IN_BATCH:
                $this->fetchRowsInBatchAndWriteThem($query);
                break;
            case self::FETCH_ROWS_ALL_AT_ONCE:
                $this->fetchAllRowsAtOnceAndWriteThem($query);
                break;
        }

        $this->reportWriter->close();
    }

    /**
     * @param string $query
     * @param int $maxFetchedRows
     * @return void
     */
    private function fetchRowsOneByOneAndWriteThem($query)
    {
        $dbRowIterator = new SingleFetchIterator($this->pdo, $query);

        foreach ($dbRowIterator as $dbRow) {
                if(!empty($dbRow['objGeofinder']) || !is_null($dbRow['objGeofinder'])){
                    $objGeofinder = json_decode($dbRow['objGeofinder'], true);
                    //$InfoCheckIn = json_decode($Venta['InfoCheckIn'], true);
                    $zonas = $objGeofinder['zonas'];
                    //ZONA
                    if(!empty($zonas[0]['valor'])){
                       $zona = $zonas[0]['valor']; 
                    }else{
                      $zona = '';  
                    }
                    //MICROZONA
                    if(!empty($zonas[1]['valor'])){
                       //$microzona = $zonas[1]['valor']; 
                       
                       $microzona = convertirMicrozona($zonas[1]['valor']);
                       
                    }else{
                      $microzona = '';  
                    }                    
                }else{
                    //ZONA
                    $zona = '';
                    //MICROZONA
                    $microzona='';
                }
            
            
            $reportRow = [$dbRow['IdVenta'],
$dbRow['IdUsuario'],
$dbRow['IdCliente'],
$dbRow['Matricula'],
$dbRow['Comercial'],
$dbRow['CC'],
$dbRow['JefeVenta'],
$dbRow['CategoriaVenta'],
$dbRow['TipoVenta'],
$dbRow['Segmento'],
$dbRow['NProspecto'],
$dbRow['NContrato'],
$dbRow['PrecioKit'],
$dbRow['TotalExtra'],
$dbRow['IngresoKIT'],
$dbRow['IngresoExtras'],
$dbRow['IngresoTotal'],
$dbRow['EstadoVenta'],
$dbRow['ExtraAutorizado'],
$dbRow['MesesAutorizado'],
$dbRow['N590'],
$dbRow['FormaPago'],
$dbRow['PAI'],
$dbRow['ResumenVenta'],
$dbRow['FechaHoraREG'],
$dbRow['EstadoVisita'],
$dbRow['FechaVisita'],
$dbRow['HoraVisita'],
$dbRow['MesVisita'],
$dbRow['FVencimientoVisita'],
$dbRow['GestorMKT'],
$dbRow['OrigenVisita'],
$dbRow['ComentarioVisita'],
$dbRow['GestorOrigen'],
$dbRow['IdGeografico'],
$dbRow['NombApellido'],
$dbRow['Telefonouno'],
$dbRow['Telefonodos'],
$dbRow['Correo'],
$dbRow['Direccion'],
$dbRow['Distrito'],
$dbRow['Provincia'],
$dbRow['CCOriginal'],
$dbRow['EstadoTelecierre'],
$dbRow['TCGestor'],
$dbRow['TCMotivoNoVenta'],
$dbRow['TCTipoVenta'],
$dbRow['Checkin'],
$dbRow['LeadID'],
$dbRow['FechaHoraInst'],
$dbRow['TipoDocumento'],
$dbRow['Documento'],
$dbRow['Accion'],
$dbRow['PerfilScoring'],
$dbRow['IdConsultaScoring'],
$dbRow['DistritalScoring'],
$dbRow['ProvincialScoring'],
$dbRow['DepartamentalScoring'],
$dbRow['MatriculaScoring'],
$dbRow['Score'],
$dbRow['PredictorIngresos_IP'],
$dbRow['CantidadDocumentosMorosos'],
$dbRow['FechaHoraConsulta'],
$zona,
$microzona,
$dbRow['IMP'],
$dbRow['pagoAdelantado']
];
            $this->reportWriter->writeRow($reportRow);
        }
    }

    /**
     * @param string $query
     * @param int $maxFetchedRows
     * @return void
     */
    private function fetchRowsInBatchAndWriteThem($query)
    {
        $idFieldName = 'IdVenta';
        $batchFetchIterator = new BatchFetchIterator($this->pdo, $query, $idFieldName, $this->batchSize);

        foreach ($batchFetchIterator as $dbRows) {
            foreach ($dbRows as $dbRow) {
                $reportRow = [$dbRow['IdVenta'], $dbRow['IdUsuario'], $dbRow['IdCliente'], $dbRow['Matricula'], $dbRow['Comercial'], $dbRow['CC']];
                $this->reportWriter->writeRow($reportRow);
            }
        }
    }

    /**
     * @param string $query
     * @return void
     */
    private function fetchAllRowsAtOnceAndWriteThem($query)
    {
        $statement = $this->pdo->prepare($query);
        $statement->execute();

        $allDBRows = $statement->fetchAll(\PDO::FETCH_ASSOC);

        foreach ($allDBRows as $dbRow) {
            $reportRow =  [$dbRow['IdVenta'], $dbRow['IdUsuario'], $dbRow['IdCliente'], $dbRow['Matricula'], $dbRow['Comercial'], $dbRow['CC']];
            $this->reportWriter->writeRow($reportRow);
        }

        $statement->closeCursor();
    }

    /**
     * @return void
     */
    private function writeReportHeader()
    {
        // The header will be bold
        $headerRow = ['IdVenta',
'IdUsuario',
'IdCliente',
'Matricula',
'Comercial',
'CC',
'JefeVenta',
'CategoriaVenta',
'TipoVenta',
'Segmento',
'NProspecto',
'NContrato',
'PrecioKit',
'TotalExtra',
'IngresoKIT',
'IngresoExtras',
'IngresoTotal',
'EstadoVenta',
'ExtraAutorizado',
'MesesAutorizado',
'N590',
'FormaPago',
'PAI',
'ResumenVenta',
'FechaHoraREG',
'EstadoVisita',
'FechaVisita',
'HoraVisita',
'MesVisita',
'FVencimientoVisita',
'GestorMKT',
'OrigenVisita',
'ComentarioVisita',
'GestorOrigen',
'IdGeografico',
'NombApellido',
'Telefonouno',
'Telefonodos',
'Correo',
'Direccion',
'Distrito',
'Provincia',
'CCOriginal',
'EstadoTelecierre',
'TCGestor',
'TCMotivoNoVenta',
'TCTipoVenta',
'Checkin',
'LeadID',
'FechaHoraInst',
'TipoDocumento',
'Documento',
'Accion',
'PerfilScoring',
'IdConsultaScoring',
'DistritalScoring',
'ProvincialScoring',
'DepartamentalScoring',
'MatriculaScoring',
'Score',
'PredictorIngresos_IP',
'CantidadDocumentosMorosos',
'FechaHoraConsulta',
'Zona',
'Seccion',
'IMP',
'pagoAdelantado'];
        $this->reportWriter->writeHeaderRow($headerRow);
    }
    //FUNCIONES DE DESCARGA DE exportTelecierreReglas
     public function fetchDataAndCreateReport_TelecierreReglas($outputPath){
           $this->reportWriter = $this->createReportWriter($outputPath);
        $this->writeReportHeader_TelecierreReglas();
        // Make sure to only select the fields we are interested in
        $query="SELECT v.IdVenta,
        v.EstadoVenta,
        v.TipoVenta,
        v.CC DC,
        u.Matricula,
        CONCAT(u.Nombres,' ',u.Apellidos) Comercial,
        v.Segmento,
        v.NProspecto,
        v.NContrato,
        c.NombApellido Cliente,
        v.CategoriaVenta,
        v.PrecioKit,
        v.FVencimientoVisita,
        v.TCGestor,
        v.EstadoTelecierre,
        v.TCComentario,
        v.FechaVisita,
        v.HoraVisita,
        v.Checkin,
        c.Telefonouno,
        c.Telefonodos,
        a.InstaladoX,
        a.FechaInst,
        a.HoraInst,
        a.FechaHoraMODIF,
        v.VisitaNoVisitable,
        v.pagoAdelantado
        FROM yifi58ge_dbVENTAS.tblventa v 
        INNER JOIN yifi58ge_dbUSUARIO.tblusuario u ON u.IdUsuario=v.IdUsuario 
        INNER JOIN yifi58ge_dbVENTAS.tblagenda a ON a.IdVenta=v.IdVenta 
        INNER JOIN yifi58ge_dbVENTAS.tblcliente c on c.IdCliente=v.IdCliente
        WHERE
        v.TipoVenta =  'RE' and v.CC<>512
        /*mostrar todas las visitas sin gestion en 24 horas y sin CHECKIN*/
        AND ((v.EstadoVenta =  'VAsignada' AND v.checkin='NO' AND  CONCAT( v.FechaVisita,' ', v.HoraVisita ) <= DATE_SUB( DATE_ADD(SYSDATE(), INTERVAL 2 HOUR) , INTERVAL 1 DAY ) )
        /*mostrar todas las visitas por conversion debajo del 20% cambio semanal sin gestion en 24 horas */
        or (v.EstadoVenta =  'VAsignada' AND ( v.CC='150' OR v.CC='151' OR v.CC='152' OR v.CC='153' OR v.CC='154' OR  v.CC='155' OR v.CC='156' OR v.CC='157' OR v.CC='158' OR v.CC='159' OR v.CC='120' OR v.CC='140' OR v.CC='40' OR v.CC='41' OR v.CC='100'  OR v.CC='200' OR v.CC='121' OR v.CC='250' OR v.CC='251' OR v.CC='80') AND  CONCAT( v.FechaVisita,' ', v.HoraVisita ) <= DATE_SUB( DATE_ADD(SYSDATE(), INTERVAL 2 HOUR) , INTERVAL 1 DAY ) )                 
        /*mostrar estadoventa=novendida*/
        or (v.EstadoVenta='NoVendida' and v.FVencimientoVisita<CURDATE())
        /*mostrar visitas con fvencimientovisita*/
        or ( v.FVencimientoVisita<=DATE_FORMAT(SYSDATE(),'%Y-%m-%d') and (v.Estadoventa ='VAsignada' or v.Estadoventa ='PendienteAgenda' or v.Estadoventa ='AgendadaTEC'))
        /*cancelada*/    
        OR (v.EstadoVenta='Cancelada' and DATE_ADD(DATE_FORMAT(a.FechaHoraMODIF,'%Y-%m-%d'), INTERVAL 1 DAY)<=DATE_FORMAT(SYSDATE(),'%Y-%m-%d')))
        OR (v.VisitaNoVisitable = 'SI' and v.Estadoventa ='VAsignada')";
        switch ($this->fetchingMethod) {
            case self::FETCH_ROWS_ONE_BY_ONE:
                $this->fetchRowsOneByOneAndWriteThem_TelecierreReglas($query);
                break;
            case self::FETCH_ROWS_IN_BATCH:
                $this->fetchRowsInBatchAndWriteThem($query);
                break;
            case self::FETCH_ROWS_ALL_AT_ONCE:
                $this->fetchAllRowsAtOnceAndWriteThem($query);
                break;
        }

        $this->reportWriter->close();
    }
    private function writeReportHeader_TelecierreReglas(){
        // The header will be bold
        $headerRow = [
            'IdVenta',
            'EstadoVenta',
            'TipoVenta',
            'DC',
            'Matricula',
            'Comercial',
            'Segmento',
            'NProspecto',
            'NContrato',
            'Cliente',
            'CategoriaVenta',
            'PrecioKit',
            'FVencimientoVisita',
            'TCGestor',
            'EstadoTelecierre',
            'TCComentario',
            'FechaVisita',
            'HoraVisita',
            'Checkin',
            'Telefonouno',
            'Telefonodos',
            'InstaladoX',
            'FechaInst',
            'HoraInst',
            'FechaHoraMODIF',
            'VisitaNoVisitable',
            'pagoAdelantado'
            ];
        $this->reportWriter->writeHeaderRow($headerRow);
    }
    
   private function fetchRowsOneByOneAndWriteThem_TelecierreReglas($query){
        $dbRowIterator = new SingleFetchIterator($this->pdo, $query);

        foreach ($dbRowIterator as $dbRow) {

        $reportRow = [
             $dbRow['IdVenta'],
             $dbRow['EstadoVenta'],
             $dbRow['TipoVenta'],
             $dbRow['DC'],
             $dbRow['Matricula'],
             $dbRow['Comercial'],
             $dbRow['Segmento'],
             $dbRow['NProspecto'],
             $dbRow['NContrato'],
             $dbRow['Cliente'],
             $dbRow['CategoriaVenta'],
             $dbRow['PrecioKit'],
             $dbRow['FVencimientoVisita'],
             $dbRow['TCGestor'],
             $dbRow['EstadoTelecierre'],
             $dbRow['TCComentario'],
             $dbRow['FechaVisita'],
             $dbRow['HoraVisita'],
             $dbRow['Checkin'],
             $dbRow['Telefonouno'],
             $dbRow['Telefonodos'],
             $dbRow['InstaladoX'],
             $dbRow['FechaInst'],
             $dbRow['HoraInst'],
             $dbRow['FechaHoraMODIF'],
             $dbRow['VisitaNoVisitable'],
             $dbRow['pagoAdelantado']
                    ];
            $this->reportWriter->writeRow($reportRow);
        }
    }     

  //FIN exportTelecierreReglas

    
}
