<?php

namespace SpoutExample;

use SpoutExample\Iterator\BatchFetchIterator;
use SpoutExample\Iterator\SingleFetchIterator;
use SpoutExample\ReportWriter\PHPExcelWriter;
use SpoutExample\ReportWriter\SpoutWriter;
use SpoutExample\ReportWriter\WriterType;

/**
 * Class ReportCreatorVisitas
 * In this example, we want to generate a report listing all the products available
 * as well as some associated data (quantity available, quantity sold, ...).
 *
 * @package SpoutExample
 */
class ReportCreatorTeleventa
{
    /** Fetching methods */
    const FETCH_ROWS_ONE_BY_ONE = 1;
    const FETCH_ROWS_IN_BATCH = 2;
    const FETCH_ROWS_ALL_AT_ONCE = 3;

    /** @var \PDO PDO instance */
    private $pdo;

    /** @var ReportWriter\WriterType Type of writer to use */
    private $writerType = WriterType::SPOUT;

    /** @var ReportWriter\AbstractWriter Writer used to create a XLSX report */
    private $reportWriter;

    /** @var int The fetching method to use, used for benchmarks */
    private $fetchingMethod = self::FETCH_ROWS_IN_BATCH;

    /** @var int Number of rows to fetch for each batch (used only when FETCH_ROWS_IN_BATCH is enabled) */
    private $batchSize;

    /**
     * @param DBConf $dbConf
     */
    public function __construct(DBConf $dbConf)
    {
        $this->pdo = new \PDO($dbConf->getDSN(), $dbConf->getUsername(), $dbConf->getPassword());
    }

    /**
     * @return self
     */
    public function setFetchRowsOneByOne()
    {
        $this->fetchingMethod = self::FETCH_ROWS_ONE_BY_ONE;
        return $this;
    }

    /**
     * @param int $batchSize Number of rows to fetch for each batch
     * @return self
     */
    public function setFetchRowsInBatch($batchSize)
    {
        $this->fetchingMethod = self::FETCH_ROWS_IN_BATCH;
        $this->batchSize = $batchSize;
        return $this;
    }

    /**
     * @return self
     */
    public function setFetchAllRowsAtOnce()
    {
        $this->fetchingMethod = self::FETCH_ROWS_ALL_AT_ONCE;
        return $this;
    }

    /**
     * @return string Name of the fetching method used
     */
    public function getFetchMethodName()
    {
        switch ($this->fetchingMethod) {
            case self::FETCH_ROWS_ONE_BY_ONE: return 'Fetch mode: one by one';
            case self::FETCH_ROWS_ALL_AT_ONCE: return 'Fetch mode: all at once';
            case self::FETCH_ROWS_IN_BATCH:
            default:
                return 'Fetch mode: batch';
        }
    }

    /**
     * @see \SpoutExample\ReportWriter\WriterType
     *
     * @param string $writerType
     * @return ReportCreator
     */
    public function setWriterType($writerType)
    {
        $this->writerType = $writerType;
        return $this;
    }

    /**
     * @param $outputPath
     * @return ReportWriter\AbstractWriter
     */
    private function createReportWriter($outputPath)
    {
        switch ($this->writerType) {
            case WriterType::PHP_EXCEL: return new PHPExcelWriter($outputPath);
            case WriterType::SPOUT:
            default:
                return new SpoutWriter($outputPath);
        }
    }

    /**
     * Fetches the data from the DB and spits it out in a XLSX file.
     *
     * @param string $outputPath Path where the report will be written to
     * @return void
     */
    public function fetchDataAndCreateReport($outputPath)
    {
        $this->reportWriter = $this->createReportWriter($outputPath);
        $this->writeReportHeader();

        // Make sure to only select the fields we are interested in
        $query = "SELECT 
v.IdVenta,
v.IdUsuario,
v.IdCliente,
v.Matricula,
v.Comercial,
v.CC,
v.JefeVenta,
v.CategoriaVenta,
v.TipoVenta,
v.Segmento,
v.NProspecto,
v.NContrato,
v.PrecioKit,
v.TotalExtra,
v.IngresoKIT,
v.IngresoExtras,
v.IngresoTotal,
v.EstadoVenta,
v.ExtraAutorizado,
v.MesesAutorizado,
v.N590,
v.FormaPago,
v.PAI,
v.ResumenVenta,
v.FechaHoraREG,
v.EstadoVisita,
v.FechaVisita,
v.HoraVisita,
v.MesVisita,
v.FVencimientoVisita,
v.GestorMKT,
v.OrigenVisita,
v.ComentarioVisita,
v.GestorOrigen,
c.IdGeografico,
c.NombApellido,
c.Telefonouno,
c.Telefonodos,
c.Correo,
c.Direccion,
g.Distrito,
g.Provincia,
v.CCOriginal,
v.EstadoTelecierre,
v.TCGestor,
v.TCMotivoNoVenta,
v.TCTipoVenta,
v.Checkin,
v.LeadID,
concat(a.FechaInst, ' ', a.HoraInst) FechaHoraInst,
d.TipoDocumento,
d.Documento,
v.Accion,
v.PerfilScoring,
v.IdConsultaScoring,
s.Distrital as DistritalScoring,
s.Provincial as ProvincialScoring,
s.Departamental as DepartamentalScoring,
s.GestorOrigen as MatriculaScoring,
s.Score,
s.PredictorIngresos_IP,
s.CantidadDocumentosMorosos,
s.FechaHoraReg as FechaHoraConsulta,
v.IMP,
c.objGeofinder,
v.pagoAdelantado
FROM yifi58ge_dbVENTAS.tblventa v INNER JOIN yifi58ge_dbVENTAS.tblcliente c on c.IdCliente=v.IdCliente
INNER JOIN yifi58ge_dbVENTAS.tblagenda a ON a.IdVenta=v.IdVenta 
INNER JOIN yifi58ge_dbVENTAS.tblgeografico g ON g.IdGeografico=c.IdGeografico
LEFT JOIN yifi58ge_dbVENTAS.tbldocumento d ON c.IdDocVisita = d.IdDoc
LEFT JOIN yifi58ge_dbVENTAS.tbltmpscoring s ON s.IdConsultaScoring=v.IdConsultaScoring    
WHERE v.TipoVenta='RE'
and v.IdVenta >= 45451";

        switch ($this->fetchingMethod) {
            case self::FETCH_ROWS_ONE_BY_ONE:
                $this->fetchRowsOneByOneAndWriteThem($query);
                break;
            case self::FETCH_ROWS_IN_BATCH:
                $this->fetchRowsInBatchAndWriteThem($query);
                break;
            case self::FETCH_ROWS_ALL_AT_ONCE:
                $this->fetchAllRowsAtOnceAndWriteThem($query);
                break;
        }

        $this->reportWriter->close();
    }

    /**
     * @param string $query
     * @param int $maxFetchedRows
     * @return void
     */
    private function fetchRowsOneByOneAndWriteThem($query)
    {
        $dbRowIterator = new SingleFetchIterator($this->pdo, $query);

        foreach ($dbRowIterator as $dbRow) {
                if(!empty($dbRow['objGeofinder']) || !is_null($dbRow['objGeofinder'])){
                    $objGeofinder = json_decode($dbRow['objGeofinder'], true);
                    //$InfoCheckIn = json_decode($Venta['InfoCheckIn'], true);
                    $zonas = $objGeofinder['zonas'];
                    //ZONA
                    if(!empty($zonas[0]['valor'])){
                       $zona = $zonas[0]['valor']; 
                    }else{
                      $zona = '';  
                    }
                    //MICROZONA
                    if(!empty($zonas[1]['valor'])){
                       //$microzona = $zonas[1]['valor']; 
                       
                       $microzona = convertirMicrozona($zonas[1]['valor']);
                       
                    }else{
                      $microzona = '';  
                    }                    
                }else{
                    //ZONA
                    $zona = '';
                    //MICROZONA
                    $microzona='';
                }
            
            
            $reportRow = [$dbRow['IdVenta'],
$dbRow['IdUsuario'],
$dbRow['IdCliente'],
$dbRow['Matricula'],
$dbRow['Comercial'],
$dbRow['CC'],
$dbRow['JefeVenta'],
$dbRow['CategoriaVenta'],
$dbRow['TipoVenta'],
$dbRow['Segmento'],
$dbRow['NProspecto'],
$dbRow['NContrato'],
$dbRow['PrecioKit'],
$dbRow['TotalExtra'],
$dbRow['IngresoKIT'],
$dbRow['IngresoExtras'],
$dbRow['IngresoTotal'],
$dbRow['EstadoVenta'],
$dbRow['ExtraAutorizado'],
$dbRow['MesesAutorizado'],
$dbRow['N590'],
$dbRow['FormaPago'],
$dbRow['PAI'],
$dbRow['ResumenVenta'],
$dbRow['FechaHoraREG'],
$dbRow['EstadoVisita'],
$dbRow['FechaVisita'],
$dbRow['HoraVisita'],
$dbRow['MesVisita'],
$dbRow['FVencimientoVisita'],
$dbRow['GestorMKT'],
$dbRow['OrigenVisita'],
$dbRow['ComentarioVisita'],
$dbRow['GestorOrigen'],
$dbRow['IdGeografico'],
$dbRow['NombApellido'],
$dbRow['Telefonouno'],
$dbRow['Telefonodos'],
$dbRow['Correo'],
$dbRow['Direccion'],
$dbRow['Distrito'],
$dbRow['Provincia'],
$dbRow['CCOriginal'],
$dbRow['EstadoTelecierre'],
$dbRow['TCGestor'],
$dbRow['TCMotivoNoVenta'],
$dbRow['TCTipoVenta'],
$dbRow['Checkin'],
$dbRow['LeadID'],
$dbRow['FechaHoraInst'],
$dbRow['TipoDocumento'],
$dbRow['Documento'],
$dbRow['Accion'],
$dbRow['PerfilScoring'],
$dbRow['IdConsultaScoring'],
$dbRow['DistritalScoring'],
$dbRow['ProvincialScoring'],
$dbRow['DepartamentalScoring'],
$dbRow['MatriculaScoring'],
$dbRow['Score'],
$dbRow['PredictorIngresos_IP'],
$dbRow['CantidadDocumentosMorosos'],
$dbRow['FechaHoraConsulta'],
$zona,
$microzona,
$dbRow['IMP'],
$dbRow['pagoAdelantado']
];

            $this->reportWriter->writeRow($reportRow);
        }
    }

    /**
     * @param string $query
     * @param int $maxFetchedRows
     * @return void
     */
    private function fetchRowsInBatchAndWriteThem($query)
    {
        $idFieldName = 'IdVenta';
        $batchFetchIterator = new BatchFetchIterator($this->pdo, $query, $idFieldName, $this->batchSize);

        foreach ($batchFetchIterator as $dbRows) {
            foreach ($dbRows as $dbRow) {
                $reportRow = [$dbRow['IdVenta'], $dbRow['IdUsuario'], $dbRow['IdCliente'], $dbRow['Matricula'], $dbRow['Comercial'], $dbRow['CC']];
                $this->reportWriter->writeRow($reportRow);
            }
        }
    }

    /**
     * @param string $query
     * @return void
     */
    private function fetchAllRowsAtOnceAndWriteThem($query)
    {
        $statement = $this->pdo->prepare($query);
        $statement->execute();

        $allDBRows = $statement->fetchAll(\PDO::FETCH_ASSOC);

        foreach ($allDBRows as $dbRow) {
            $reportRow =  [$dbRow['IdVenta'], $dbRow['IdUsuario'], $dbRow['IdCliente'], $dbRow['Matricula'], $dbRow['Comercial'], $dbRow['CC']];
            $this->reportWriter->writeRow($reportRow);
        }

        $statement->closeCursor();
    }

    /**
     * @return void
     */
    private function writeReportHeader()
    {
        // The header will be bold
        $headerRow = ['IdVenta',
'IdUsuario',
'IdCliente',
'Matricula',
'Comercial',
'CC',
'JefeVenta',
'CategoriaVenta',
'TipoVenta',
'Segmento',
'NProspecto',
'NContrato',
'PrecioKit',
'TotalExtra',
'IngresoKIT',
'IngresoExtras',
'IngresoTotal',
'EstadoVenta',
'ExtraAutorizado',
'MesesAutorizado',
'N590',
'FormaPago',
'PAI',
'ResumenVenta',
'FechaHoraREG',
'EstadoVisita',
'FechaVisita',
'HoraVisita',
'MesVisita',
'FVencimientoVisita',
'GestorMKT',
'OrigenVisita',
'ComentarioVisita',
'GestorOrigen',
'IdGeografico',
'NombApellido',
'Telefonouno',
'Telefonodos',
'Correo',
'Direccion',
'Distrito',
'Provincia',
'CCOriginal',
'EstadoTelecierre',
'TCGestor',
'TCMotivoNoVenta',
'TCTipoVenta',
'Checkin',
'LeadID',
'FechaHoraInst',
'TipoDocumento',
'Documento',
'Accion',
'PerfilScoring',
'IdConsultaScoring',
'DistritalScoring',
'ProvincialScoring',
'DepartamentalScoring',
'MatriculaScoring',
'Score',
'PredictorIngresos_IP',
'CantidadDocumentosMorosos',
'FechaHoraConsulta',
'Zona',
'Seccion',
'IMP'];
        $this->reportWriter->writeHeaderRow($headerRow);
    }
    //FUNCIONES DE DESCARGA DE exportTeleventa
     public function fetchDataAndCreateReport_Televenta($outputPath){
           $this->reportWriter = $this->createReportWriter($outputPath);
        $this->writeReportHeader_Televenta();
        // Make sure to only select the fields we are interested in
        $query="SELECT
        v.IdVenta AS IdVenta,
        c.IdCliente AS IdCliente,
        c.Telefonouno AS Telefonouno,
        c.Telefonodos AS Telefonodos,
        c.NombApellido AS Cliente,
        v.EstadoTelecierre AS EstadoTeleventa,
        v.EstadoVenta AS EstadoVenta,
        v.TipoVenta AS TipoVenta,
        v.FechaHoraREG AS FechaHoraREG,
        v.TipoTV AS TipoTV,
        v.TCGestor AS TCGestor,
        CONCAT(u.Nombres, ' ', u.Apellidos) AS NombreTV,
        v.Segmento AS Segmento,
        v.PrecioKit AS PrecioKitTV,
        v.N590 AS N590,
        de.Departamento AS Departamento,
        p.Provincia AS Provincia,
        g.Distrito AS Distrito,
        d.Zonas AS Zonas,
        v.Vueltas AS Vueltas,
        v.OrigenVisita AS OrigenVisita,
        o.Canal AS Canal,
        a.FechaInst AS FechaInst,
        a.HoraInst AS HoraInst,
        v.CC AS CC,
        v.Matricula AS MatriculaTVOriginal,
        v.NProspecto AS NProspecto,
        v.NContrato AS NContrato,
        v.IdLead AS IdLead,
        l.TipoTransferencia,
        v.TCFechaRellamada as FechaRellamadaTV,
        v.TCHoraRellamada as HoraRellamadaTV,
        v.Nivel_1 as Contacto,
        v.Nivel_2 as Gestionable,
        v.TCMotivoNoVenta as SubEstadoTeleventa,
        v.TCCompetencia as SubCompetencia,
		uGest.Matricula as ultimoGestorCC,
        Concat(uGest.Nombres,' ',uGest.Apellidos) as ultimoNombreGestorCC,
        l.ObsTransferencia,
        v.FechaUltimaGestion,
        l.FechaUltimaGestion as FechaUltimaGestionCC,
        l.LeadID,
        a.Repercusion,
        a.Comentario as ComentarioAgenda,
        v.TipoKIT,
        v.TotalExtra,
        v.CantidadExtras as CantidadExtrasTV,
        v.TCComentario as ComentarioTV,
        v.ComentarioCancelacion,
        ref.TipoReferido,
        v.referidoHabilitadoTVTC,
        v.TotalCuotaMonitoreo,
        v.CantidadRegalo,
        v.DescMeses,
        v.PAI,
        v.TCScoring,
        t.PerfilScoring,
        f.FechaRecepcionDOC,
        f.FechaFacturacion,
        a.RepercusionCosto,
        a.Desplazamiento,
        v.ResumenVenta,
        v.ProspectoAbierto,
        v.FechaProspectoAbierto,
        v.pagoAdelantado,
        l.conSistema,
        l.empresaSistema,
        v.TCTipoVenta,
        l.ResultadoGestion as ResultadoGestionCC,
        gl.SubTipificacion as SubtipificacionCC,
        l.FechaUltimaGestionRemarketing as FechaUltimaGestionRemarketingCC,
        l.ResultadoGestionRemarketing as ResultadoGestionRemarketingCC,
        glr.SubTipificacion as SubtipificacionRemarketingCC,
        v.CCTVTC,
        ge.vueltasTeleventa,
        ge.ultimoGestorTeleventa,
        ge.ResultadoGestionTeleventa,
        ge.SubtipificacionTeleventa,
        ge.MotivoSubtipificacionTeleventa,
        v.FechaHoraTransferenciaTCTeleventa,
        ge.vueltasTCTeleventa,
        ge.ultimoGestorTCTeleventa,
        ge.ResultadoGestionTCTeleventa,
        ge.SubtipificacionTCTeleventa,
        ge.MotivoSubtipificacionTCTeleventa,
        v.FechaHoraTransferenciaTVRemarketing,
        ge.vueltasTVRemarketing,
        ge.ultimoGestorTVRemarketing,
        ge.ResultadoGestionTVRemarketing,
        ge.SubtipificacionTVRemarketing,
        ge.MotivoSubtipificacionTVRemarketing
    FROM
     yifi58ge_dbVENTAS.tblventa v
     JOIN yifi58ge_dbVENTAS.tblcliente c ON c.IdCliente = v.IdCliente
     JOIN yifi58ge_dbVENTAS.tblagenda a ON a.IdVenta = v.IdVenta
     JOIN yifi58ge_dbVENTAS.tblfacturacion f ON f.IdVenta = v.IdVenta
     JOIN yifi58ge_dbUSUARIO.tblusuario u ON u.Matricula = v.TCGestor
     LEFT JOIN yifi58ge_dbVENTAS.tblgeografico g ON g.IdGeografico = c.IdGeografico
     LEFT JOIN yifi58ge_dbVENTAS.tbldistrital d ON d.CodUbigeoReniec = g.CodUbigeoReniec
     LEFT JOIN yifi58ge_dbVENTAS.tblprovincial p ON p.IdProvincial = d.IdProvincial
     LEFT JOIN yifi58ge_dbVENTAS.tbldepartamental de ON de.IdDepartamental = p.IdDepartamental
     LEFT JOIN yifi58ge_dbVENTAS.tblorigen o ON o.Origen = v.OrigenVisita
     LEFT JOIN yifi58ge_dbVENTAS.tblLeadCC l ON l.IdLeadCC = v.IdLead
     LEFT JOIN yifi58ge_dbVENTAS.tblGestionLead gl on gl.IdGestionLead=l.UltimaGestion
     LEFT JOIN yifi58ge_dbUSUARIO.tblusuario uGest on gl.IdUsuario = uGest.IdUsuario
     LEFT JOIN yifi58ge_dbREFERIDO.tblreferido_detalleTCTV ref on ref.IdVenta = v.IdVenta
     LEFT JOIN yifi58ge_dbVENTAS.tbltmpscoring t on t.IdConsultaScoring = v.IdConsultaScoringTCTV
     LEFT JOIN yifi58ge_dbVENTAS.tblTCTVgestion ge on ge.id = v.IdTCTVgestion
     LEFT JOIN yifi58ge_dbVENTAS.tblGestionLead glr on glr.IdGestionLead=l.UltimaGestionRemarketing
    WHERE
     v.CC = 512";
        switch ($this->fetchingMethod) {
            case self::FETCH_ROWS_ONE_BY_ONE:
                $this->fetchRowsOneByOneAndWriteThem_Televenta($query);
                break;
            case self::FETCH_ROWS_IN_BATCH:
                $this->fetchRowsInBatchAndWriteThem($query);
                break;
            case self::FETCH_ROWS_ALL_AT_ONCE:
                $this->fetchAllRowsAtOnceAndWriteThem($query);
                break;
        }

        $this->reportWriter->close();
    }
    private function writeReportHeader_Televenta(){
        // The header will be bold
        $headerRow = [
            'IdVenta',
            'IdCliente',
            'Telefonouno',
            'Telefonodos',
            'Cliente',
            'EstadoTeleventa',
            'EstadoVenta',
            'TipoVenta',
            'FechaHoraREG',
            'TipoTV',
            'TCGestor',
            'NombreTV',
            'Segmento',
            'PrecioKitTV',
            'N590',
            'Departamento',
            'Provincia',
            'Distrito',
            'Zonas',
            'Vueltas',
            'OrigenVisita',
            'Canal',
            'FechaInst',
            'HoraInst',
            'CC',
            'MatriculaTVOriginal',
            'NProspecto',
            'NContrato',
            'IdLead',
            'TipoTransferencia',
            'FechaRellamadaTV',
            'HoraRellamadaTV',
            'Contacto',
            'Gestionable',
            'SubEstadoTeleventa',
            'SubCompetencia',
            'ultimoGestorCC',
            'ultimoNombreGestorCC',
            'ObsTransferencia',
            'FechaUltimaGestion',
            'FechaUltimaGestionCC',
            'LeadID',
            'Repercusion',
            'ComentarioAgenda',
            'TipoKIT',
            'TotalExtra',
            'CantidadExtrasTV',
            'ComentarioTV',
            'ComentarioCancelacion',
            'TipoReferido',
            'referidoHabilitadoTVTC',
            'TotalCuotaMonitoreo',
            'CantidadRegalo',
            'DescMeses',
            'PAI',
            'TCScoring',
            'PerfilScoring',
            'FechaRecepcionDOC',
            'FechaFacturacion',
            'RepercusionCosto',
            'Desplazamiento',
            'ResumenVenta',
            'ProspectoAbierto',
            'FechaProspectoAbierto',
            'pagoAdelantado',
            'conSistema',
            'empresaSistema',
            'TCTipoVenta',
            'ResultadoGestionCC',
            'SubtipificacionCC',
            'FechaUltimaGestionRemarketingCC',
            'ResultadoGestionRemarketingCC',
            'SubtipificacionRemarketingCC',
            'CCTVTC',
            'vueltasTeleventa',
            'ultimoGestorTeleventa',
            'ResultadoGestionTeleventa',
            'SubtipificacionTeleventa',
            'MotivoSubtipificacionTeleventa',
            'FechaHoraTransferenciaTCTeleventa',
            'vueltasTCTeleventa',
            'ultimoGestorTCTeleventa',
            'ResultadoGestionTCTeleventa',
            'SubtipificacionTCTeleventa',
            'MotivoSubtipificacionTCTeleventa',
            'FechaHoraTransferenciaTVRemarketing',
            'vueltasTVRemarketing',
            'ultimoGestorTVRemarketing',
            'ResultadoGestionTVRemarketing',
            'SubtipificacionTVRemarketing',
            'MotivoSubtipificacionTVRemarketing'
            ];
        $this->reportWriter->writeHeaderRow($headerRow);
    }
    
   private function fetchRowsOneByOneAndWriteThem_Televenta($query){
        $dbRowIterator = new SingleFetchIterator($this->pdo, $query);

        foreach ($dbRowIterator as $dbRow) {

        $reportRow = [
            $dbRow['IdVenta'],
            $dbRow['IdCliente'],
            $dbRow['Telefonouno'],
            $dbRow['Telefonodos'],
            $dbRow['Cliente'],
            $dbRow['EstadoTeleventa'],
            $dbRow['EstadoVenta'],
            $dbRow['TipoVenta'],
            $dbRow['FechaHoraREG'],
            $dbRow['TipoTV'],
            $dbRow['TCGestor'],
            $dbRow['NombreTV'],
            $dbRow['Segmento'],
            $dbRow['PrecioKitTV'],
            $dbRow['N590'],
            $dbRow['Departamento'],
            $dbRow['Provincia'],
            $dbRow['Distrito'],
            $dbRow['Zonas'],
            $dbRow['Vueltas'],
            $dbRow['OrigenVisita'],
            $dbRow['Canal'],
            $dbRow['FechaInst'],
            $dbRow['HoraInst'],
            $dbRow['CC'],
            $dbRow['MatriculaTVOriginal'],
            $dbRow['NProspecto'],
            $dbRow['NContrato'],
            $dbRow['IdLead'],
            $dbRow['TipoTransferencia'],
            $dbRow['FechaRellamadaTV'],
            $dbRow['HoraRellamadaTV'],
            $dbRow['Contacto'],
            $dbRow['Gestionable'],
            $dbRow['SubEstadoTeleventa'],
            $dbRow['SubCompetencia'],
            $dbRow['ultimoGestorCC'],
            $dbRow['ultimoNombreGestorCC'],
            $dbRow['ObsTransferencia'],
            $dbRow['FechaUltimaGestion'],
            $dbRow['FechaUltimaGestionCC'],
            $dbRow['LeadID'],
            $dbRow['Repercusion'],
            $dbRow['ComentarioAgenda'],
            $dbRow['TipoKIT'],
            $dbRow['TotalExtra'],
            $dbRow['CantidadExtrasTV'],
            $dbRow['ComentarioTV'],
            $dbRow['ComentarioCancelacion'],
            $dbRow['TipoReferido'],
            $dbRow['referidoHabilitadoTVTC'],
            $dbRow['TotalCuotaMonitoreo'],
            $dbRow['CantidadRegalo'],
            $dbRow['DescMeses'],
            $dbRow['PAI'],
            $dbRow['TCScoring'],
            $dbRow['PerfilScoring'],
            $dbRow['FechaRecepcionDOC'],
            $dbRow['FechaFacturacion'],
            $dbRow['RepercusionCosto'],
            $dbRow['Desplazamiento'],
            $dbRow['ResumenVenta'],
            $dbRow['ProspectoAbierto'],
            $dbRow['FechaProspectoAbierto'],
            $dbRow['pagoAdelantado'],
            $dbRow['conSistema'],
            $dbRow['empresaSistema'],
            $dbRow['TCTipoVenta'],
            $dbRow['ResultadoGestionCC'],
            $dbRow['SubtipificacionCC'],
            $dbRow['FechaUltimaGestionRemarketingCC'],
            $dbRow['ResultadoGestionRemarketingCC'],
            $dbRow['SubtipificacionRemarketingCC'],            
            $dbRow['CCTVTC'],
            $dbRow['vueltasTeleventa'],
            $dbRow['ultimoGestorTeleventa'],
            $dbRow['ResultadoGestionTeleventa'],
            $dbRow['SubtipificacionTeleventa'],
            $dbRow['MotivoSubtipificacionTeleventa'],
            $dbRow['FechaHoraTransferenciaTCTeleventa'],
            $dbRow['vueltasTCTeleventa'],
            $dbRow['ultimoGestorTCTeleventa'],
            $dbRow['ResultadoGestionTCTeleventa'],
            $dbRow['SubtipificacionTCTeleventa'],
            $dbRow['MotivoSubtipificacionTCTeleventa'],
            $dbRow['FechaHoraTransferenciaTVRemarketing'],
            $dbRow['vueltasTVRemarketing'],
            $dbRow['ultimoGestorTVRemarketing'],
            $dbRow['ResultadoGestionTVRemarketing'],
            $dbRow['SubtipificacionTVRemarketing'],
            $dbRow['MotivoSubtipificacionTVRemarketing']
                    ];
            $this->reportWriter->writeRow($reportRow);
        }
    }     

  //FIN exportTeleventa
    //FUNCIONES TV ASIGNACION
     public function fetchDataAndCreateReport_TeleventaLead($outputPath){
           $this->reportWriter = $this->createReportWriter($outputPath);
        $this->writeReportHeader_TeleventaLead();
        // Make sure to only select the fields we are interested in
        $query="SELECT 
    l.IdLeadCC, 
    l.FechaHoraReg, 
    l.Telefono, 
    l.Gestion, 
    l.LeadID, 
    l.ResultadoGestion, 
    u.Matricula, 
    l.Nombres, 
    l.Apellidos, 
    l.Campaña, 
    l.Telefono2, 
    l.LeadID, 
    l.Observaciones, 
    l.Segmento, 
    CONCAT(l.Apellidos, ' ', l.Nombres) NombreCompleto,
    l.Vueltas,
    l.Origen,
    l.Departamento,
    l.Provincia,
    l.Distrito,
    s.Accion,
    l.FechaHoraCotizacion,
    gl.Contacto,
    gl.SubTipificacion,
    uGest.Matricula GestorUltimo,
	l.TipoTransferencia,
	l.GestorTV,
	l.EstadoLead,
	l.FechaHoraAsignacion,
	l.FechaUltimaGestion,
	l.ObsTransferencia
    FROM tblLeadCC l
    INNER JOIN yifi58ge_dbUSUARIO.tblusuario u ON l.idUsuario = u.idUsuario
    LEFT JOIN tbltmpscoring s ON l.IdScoring = s.IdConsultaScoring
    LEFT JOIN tblGestionLead gl ON l.UltimaGestion = gl.IdGestionLead
    LEFT JOIN yifi58ge_dbUSUARIO.tblusuario uGest ON gl.idUsuario = uGest.idUsuario
	WHERE (l.TipoTransferencia='BD' or l.TipoTransferencia='BD48horas') and (l.EstadoLead='' or l.EstadoLead IS NULL)";
        switch ($this->fetchingMethod) {
            case self::FETCH_ROWS_ONE_BY_ONE:
                $this->fetchRowsOneByOneAndWriteThem_TeleventaLead($query);
                break;
            case self::FETCH_ROWS_IN_BATCH:
                $this->fetchRowsInBatchAndWriteThem($query);
                break;
            case self::FETCH_ROWS_ALL_AT_ONCE:
                $this->fetchAllRowsAtOnceAndWriteThem($query);
                break;
        }

        $this->reportWriter->close();
    }
    private function writeReportHeader_TeleventaLead(){
        // The header will be bold
        $headerRow = [
            'IdLeadCC',
            'FechaHoraReg',
            'Telefono',
            'Gestion',
            'LeadID',
            'ResultadoGestion',
            'Matricula',
            'Nombres',
            'Apellidos',
            'Campaña',
            'Telefono2',
            'LeadID',
            'Observaciones',
            'Segmento',
            'NombreCompleto',
            'Vueltas',
            'Origen',
            'Departamento',
            'Provincia',
            'Distrito',
            'Accion',
            'FechaHoraCotizacion',
            'Contacto',
            'SubTipificacion',
            'GestorUltimo',
            'TipoTransferencia',
            'GestorTV',
            'EstadoLead',
	        'FechaHoraAsignacion',
	        'FechaUltimaGestion',
	        'ObsTransferencia'
            ];
        $this->reportWriter->writeHeaderRow($headerRow);
    }
    
   private function fetchRowsOneByOneAndWriteThem_TeleventaLead($query){
        $dbRowIterator = new SingleFetchIterator($this->pdo, $query);

        foreach ($dbRowIterator as $dbRow) {

        $reportRow = [
            $dbRow['IdLeadCC'],
            $dbRow['FechaHoraReg'],
            $dbRow['Telefono'],
            $dbRow['Gestion'],
            $dbRow['LeadID'],
            $dbRow['ResultadoGestion'],
            $dbRow['Matricula'],
            $dbRow['Nombres'],
            $dbRow['Apellidos'],
            $dbRow['Campaña'],
            $dbRow['Telefono2'],
            $dbRow['LeadID'],
            $dbRow['Observaciones'],
            $dbRow['Segmento'],
            $dbRow['NombreCompleto'],
            $dbRow['Vueltas'],
            $dbRow['Origen'],
            $dbRow['Departamento'],
            $dbRow['Provincia'],
            $dbRow['Distrito'],
            $dbRow['Accion'],
            $dbRow['FechaHoraCotizacion'],
            $dbRow['Contacto'],
            $dbRow['SubTipificacion'],
            $dbRow['GestorUltimo'],
            $dbRow['TipoTransferencia'],
            $dbRow['GestorTV'],
            $dbRow['EstadoLead'],
            $dbRow['FechaHoraAsignacion'],
            $dbRow['FechaUltimaGestion'],
            $dbRow['ObsTransferencia']
                    ];
            $this->reportWriter->writeRow($reportRow);
        }
    }     

  //FIN exportTeleventaLead

    //FUNCIONES TV COMISIOTRON
     public function fetchDataAndCreateReport_TeleventaComisiotron($outputPath){
           $this->reportWriter = $this->createReportWriter($outputPath);
        $this->writeReportHeader_TeleventaComisiotron();
        // Make sure to only select the fields we are interested in
        $query="SELECT v.IdVenta,
v.EstadoVenta,
v.EstadoVisita,
v.CC,
v.CCOriginal,
v.Matricula,
v.Comercial,
v.FechaVisita,
v.FVencimientoVisita,
IFNULL(STR_TO_DATE(CONCAT(v.N590,',01'),'%b-%Y,%d'),'') N590,
v.Checkin,
v.Segmento,
v.NProspecto,
v.NContrato,
c.NombApellido as Cliente,
v.EstadoTelecierre,
v.TCGestor,
v.TCMotivoNoventa,
v.TCComentario,
v.TCTipoVenta,
v.PrecioKit,
v.TotalExtra,
a.InstaladoX,
a.FechaInst,
a.HoraInst,
v.Vueltas as Contactividad,
v.TCVisitado,
v.TCScoring,
v.TCCompetencia,
IFNULL(STR_TO_DATE(CONCAT(v.MesVisita,',01'),'%b-%Y,%d'),'') MesVisita,
v.FechaHoraREG,
v.NProspectoCC,
v.TipoTV,
c.Telefonouno,
c.Telefonodos,
g.Provincia,
g.Distrito,
IFNULL(STR_TO_DATE(CONCAT(v.MesTeleventa,',01'),'%b-%Y,%d'),'') MesTeleventa,
v.LeadID,
v.IMP,
v.DescMeses,
v.OrigenVisita as Origen,
d.TipoDocumento TipoDocumentoTV,
d.Documento DocumentoTV,
d2.TipoDocumento TipoDocumentoTC,
d2.Documento DocumentoTC,
v.CantidadExtras,
v.ExtraAutorizado,
v.TipoKIT,
v.TipoVenta,
v.Prioridad,
v.pagoAdelantado,
v.FechaUltimaGestion
FROM yifi58ge_dbVENTAS.tblventa v 
INNER JOIN yifi58ge_dbUSUARIO.tblusuario u ON u.IdUsuario=v.IdUsuario 
INNER JOIN yifi58ge_dbVENTAS.tblagenda a ON a.IdVenta=v.IdVenta 
INNER JOIN yifi58ge_dbVENTAS.tblcliente c on c.IdCliente=v.IdCliente
INNER JOIN yifi58ge_dbVENTAS.tblgeografico g on g.IdGeografico=c.IdGeografico
LEFT JOIN tbldocumento d on d.IdDoc=c.IdDocTV
LEFT JOIN tbldocumento d2 on d2.IdDoc=c.IdDocTC 
WHERE v.TipoVenta =  'RE'
and (v.N590 = 'Jan-2022' or v.N590 = 'Feb-2022' or v.N590 = 'Mar-2022' or v.N590 = 'Apr-2022' or v.N590 = 'May-2022' or v.N590 = 'Jun-2022' or v.N590 = 'Jul-2022' or v.N590 = 'Aug-2022' or v.N590 = 'Sep-2022' or v.N590 = 'Oct-2022' or v.N590 = 'Nov-2022' or v.N590 = 'Dec-2022' or
v.N590 = 'Jan-2023' or v.N590 = 'Feb-2023' or v.N590 = 'Mar-2023' or v.N590 = 'Apr-2023' or v.N590 = 'May-2023' or v.N590 = 'Jun-2023' or v.N590 = 'Jul-2023' or v.N590 = 'Aug-2023' or v.N590 = 'Sep-2023' or v.N590 = 'Oct-2023' or v.N590 = 'Nov-2023' or v.N590 = 'Dec-2023')
and (v.CC='512' or v.CC='511')";
        switch ($this->fetchingMethod) {
            case self::FETCH_ROWS_ONE_BY_ONE:
                $this->fetchRowsOneByOneAndWriteThem_TeleventaComisiotron($query);
                break;
            case self::FETCH_ROWS_IN_BATCH:
                $this->fetchRowsInBatchAndWriteThem($query);
                break;
            case self::FETCH_ROWS_ALL_AT_ONCE:
                $this->fetchAllRowsAtOnceAndWriteThem($query);
                break;
        }

        $this->reportWriter->close();
    }
    private function writeReportHeader_TeleventaComisiotron(){
        // The header will be bold
        $headerRow = [
'IdVenta',
'EstadoVenta',
'EstadoVisita',
'CC',
'CCOriginal',
'Matricula',
'Comercial',
'FechaVisita',
'FVencimientoVisita',
'N590',
'Checkin',
'Segmento',
'NProspecto',
'NContrato',
'Cliente',
'EstadoTelecierre',
'TCGestor',
'TCMotivoNoventa',
'TCComentario',
'TCTipoVenta',
'PrecioKit',
'TotalExtra',
'InstaladoX',
'FechaInst',
'HoraInst',
'Contactividad',
'TCVisitado',
'TCScoring',
'TCCompetencia',
'MesVisita',
'FechaHoraREG',
'NProspectoCC',
'TipoTV',
'Telefonouno',
'Telefonodos',
'Provincia',
'Distrito',
'MesTeleventa',
'LeadID',
'IMP',
'Prioridad',
'TipoDocumentoTV',
'DocumentoTV',
'TipoDocumentoTC',
'DocumentoTC',
'DescMeses',
'CantidadExtras',
'ExtraAutorizado',
'TipoKIT',
'TipoVenta',
'Origen',
'pagoAdelantado',
'FechaUltimaGestion'
];
        $this->reportWriter->writeHeaderRow($headerRow);
    }
    
   private function fetchRowsOneByOneAndWriteThem_TeleventaComisiotron($query){
        $dbRowIterator = new SingleFetchIterator($this->pdo, $query);

        foreach ($dbRowIterator as $dbRow) {

        $reportRow = [
            $dbRow['IdVenta'],
            $dbRow['EstadoVenta'],
            $dbRow['EstadoVisita'],
            $dbRow['CC'],
            $dbRow['CCOriginal'],
            $dbRow['Matricula'],
            $dbRow['Comercial'],
            $dbRow['FechaVisita'],
            $dbRow['FVencimientoVisita'],
            $dbRow['N590'],
            $dbRow['Checkin'],
            $dbRow['Segmento'],
            $dbRow['NProspecto'],
            $dbRow['NContrato'],
            $dbRow['Cliente'],
            $dbRow['EstadoTelecierre'],
            $dbRow['TCGestor'],
            $dbRow['TCMotivoNoventa'],
            $dbRow['TCComentario'],
            $dbRow['TCTipoVenta'],
            $dbRow['PrecioKit'],
            $dbRow['TotalExtra'],
            $dbRow['InstaladoX'],
            $dbRow['FechaInst'],
            $dbRow['HoraInst'],
            $dbRow['Contactividad'],
            $dbRow['TCVisitado'],
            $dbRow['TCScoring'],
            $dbRow['TCCompetencia'],
            $dbRow['MesVisita'],
            $dbRow['FechaHoraREG'],
            $dbRow['NProspectoCC'],
            $dbRow['TipoTV'],
            $dbRow['Telefonouno'],
            $dbRow['Telefonodos'],
            $dbRow['Provincia'],
            $dbRow['Distrito'],
            $dbRow['MesTeleventa'],
            $dbRow['LeadID'],
            $dbRow['IMP'],
            $dbRow['Prioridad'],
            $dbRow['TipoDocumentoTV'],
            $dbRow['DocumentoTV'],
            $dbRow['TipoDocumentoTC'],
            $dbRow['DocumentoTC'],
            $dbRow['DescMeses'],
            $dbRow['CantidadExtras'],
            $dbRow['ExtraAutorizado'],
            $dbRow['TipoKIT'],
            $dbRow['TipoVenta'],
            $dbRow['Origen'],
            $dbRow['pagoAdelantado'],
            $dbRow['FechaUltimaGestion']
                    ];
            $this->reportWriter->writeRow($reportRow);
        }
    }     

  //FIN exportTeleventaLead
  
    //FUNCIONES DE DESCARGA DE exportTCTeleventa
     public function fetchDataAndCreateReport_TCTeleventa($outputPath){
           $this->reportWriter = $this->createReportWriter($outputPath);
        $this->writeReportHeader_TCTeleventa();
        // Make sure to only select the fields we are interested in
        $query="SELECT
        v.IdVenta AS IdVenta,
        c.IdCliente AS IdCliente,
        c.Telefonouno AS Telefonouno,
        c.Telefonodos AS Telefonodos,
        c.NombApellido AS Cliente,
        v.EstadoTelecierre AS EstadoTeleventa,
        v.EstadoVenta AS EstadoVenta,
        v.TipoVenta AS TipoVenta,
        v.FechaHoraREG AS FechaHoraREG,
        v.TipoTV AS TipoTV,
        v.TCGestor AS TCGestor,
        CONCAT(u.Nombres, ' ', u.Apellidos) AS NombreTV,
        v.Segmento AS Segmento,
        v.PrecioKit AS PrecioKitTV,
        v.N590 AS N590,
        de.Departamento AS Departamento,
        p.Provincia AS Provincia,
        g.Distrito AS Distrito,
        d.Zonas AS Zonas,
        v.Vueltas AS Vueltas,
        v.OrigenVisita AS OrigenVisita,
        o.Canal AS Canal,
        a.FechaInst AS FechaInst,
        a.HoraInst AS HoraInst,
        v.CC AS CC,
        v.Matricula AS MatriculaTVOriginal,
        v.NProspecto AS NProspecto,
        v.NContrato AS NContrato,
        v.IdLead AS IdLead,
        l.TipoTransferencia,
        v.TCFechaRellamada as FechaRellamadaTV,
        v.TCHoraRellamada as HoraRellamadaTV,
        v.Nivel_1 as Contacto,
        v.Nivel_2 as Gestionable,
        v.TCMotivoNoVenta as SubEstadoTeleventa,
        v.TCCompetencia as SubCompetencia,
		uGest.Matricula as ultimoGestorCC,
        Concat(uGest.Nombres,' ',uGest.Apellidos) as ultimoNombreGestorCC,
        l.ObsTransferencia,
        v.FechaUltimaGestion,
        l.FechaUltimaGestion as FechaUltimaGestionCC,
        l.LeadID,
        a.Repercusion,
        a.Comentario as ComentarioAgenda,
        v.TipoKIT,
        v.TotalExtra,
        v.CantidadExtras as CantidadExtrasTV,
        v.TCComentario as ComentarioTV,
        v.ComentarioCancelacion,
        ref.TipoReferido,
        v.referidoHabilitadoTVTC,
        v.TotalCuotaMonitoreo,
        v.CantidadRegalo,
        v.DescMeses,
        v.PAI,
        v.TCScoring,
        t.PerfilScoring,
        f.FechaRecepcionDOC,
        f.FechaFacturacion,
        a.RepercusionCosto,
        a.Desplazamiento,
        v.ResumenVenta,
        v.ProspectoAbierto,
        v.FechaProspectoAbierto,
        v.pagoAdelantado,
        l.conSistema,
        l.empresaSistema,
        ge.vueltasTeleventa,
        ge.ultimoGestorTeleventa,
        ge.ResultadoGestionTeleventa,
        ge.SubtipificacionTeleventa,
        ge.MotivoSubtipificacionTeleventa,
        v.FechaHoraTransferenciaTCTeleventa,
        ge.vueltasTCTeleventa,
        ge.ultimoGestorTCTeleventa,
        ge.ResultadoGestionTCTeleventa,
        ge.SubtipificacionTCTeleventa,
        ge.MotivoSubtipificacionTCTeleventa
    FROM
     yifi58ge_dbVENTAS.tblventa v
     JOIN yifi58ge_dbVENTAS.tblcliente c ON c.IdCliente = v.IdCliente
     JOIN yifi58ge_dbVENTAS.tblagenda a ON a.IdVenta = v.IdVenta
     JOIN yifi58ge_dbVENTAS.tblfacturacion f ON f.IdVenta = v.IdVenta
     JOIN yifi58ge_dbUSUARIO.tblusuario u ON u.Matricula = v.TCGestor
     LEFT JOIN yifi58ge_dbVENTAS.tblgeografico g ON g.IdGeografico = c.IdGeografico
     LEFT JOIN yifi58ge_dbVENTAS.tbldistrital d ON d.CodUbigeoReniec = g.CodUbigeoReniec
     LEFT JOIN yifi58ge_dbVENTAS.tblprovincial p ON p.IdProvincial = d.IdProvincial
     LEFT JOIN yifi58ge_dbVENTAS.tbldepartamental de ON de.IdDepartamental = p.IdDepartamental
     LEFT JOIN yifi58ge_dbVENTAS.tblorigen o ON o.Origen = v.OrigenVisita
     LEFT JOIN yifi58ge_dbVENTAS.tblLeadCC l ON l.IdLeadCC = v.IdLead
     LEFT JOIN yifi58ge_dbVENTAS.tblGestionLead gl on gl.IdGestionLead=l.UltimaGestion
     LEFT JOIN yifi58ge_dbUSUARIO.tblusuario uGest on gl.IdUsuario = uGest.IdUsuario
     LEFT JOIN yifi58ge_dbREFERIDO.tblreferido_detalleTCTV ref on ref.IdVenta = v.IdVenta
     LEFT JOIN yifi58ge_dbVENTAS.tbltmpscoring t on t.IdConsultaScoring = v.IdConsultaScoringTCTV
     LEFT JOIN yifi58ge_dbVENTAS.tblTCTVgestion ge on ge.id = v.IdTCTVgestion
    WHERE
     v.CC = 512 and v.TCTipoVenta='TCTeleventa'";
        switch ($this->fetchingMethod) {
            case self::FETCH_ROWS_ONE_BY_ONE:
                $this->fetchRowsOneByOneAndWriteThem_TCTeleventa($query);
                break;
            case self::FETCH_ROWS_IN_BATCH:
                $this->fetchRowsInBatchAndWriteThem($query);
                break;
            case self::FETCH_ROWS_ALL_AT_ONCE:
                $this->fetchAllRowsAtOnceAndWriteThem($query);
                break;
        }

        $this->reportWriter->close();
    }
    private function writeReportHeader_TCTeleventa(){
        // The header will be bold
        $headerRow = [
            'IdVenta',
            'IdCliente',
            'Telefonouno',
            'Telefonodos',
            'Cliente',
            'EstadoTeleventa',
            'EstadoVenta',
            'TipoVenta',
            'FechaHoraREG',
            'TipoTV',
            'TCGestor',
            'NombreTV',
            'Segmento',
            'PrecioKitTV',
            'N590',
            'Departamento',
            'Provincia',
            'Distrito',
            'Zonas',
            'Vueltas',
            'OrigenVisita',
            'Canal',
            'FechaInst',
            'HoraInst',
            'CC',
            'MatriculaTVOriginal',
            'NProspecto',
            'NContrato',
            'IdLead',
            'TipoTransferencia',
            'FechaRellamadaTV',
            'HoraRellamadaTV',
            'Contacto',
            'Gestionable',
            'SubEstadoTeleventa',
            'SubCompetencia',
            'ultimoGestorCC',
            'ultimoNombreGestorCC',
            'ObsTransferencia',
            'FechaUltimaGestion',
            'FechaUltimaGestionCC',
            'LeadID',
            'Repercusion',
            'ComentarioAgenda',
            'TipoKIT',
            'TotalExtra',
            'CantidadExtrasTV',
            'ComentarioTV',
            'ComentarioCancelacion',
            'TipoReferido',
            'referidoHabilitadoTVTC',
            'TotalCuotaMonitoreo',
            'CantidadRegalo',
            'DescMeses',
            'PAI',
            'TCScoring',
            'PerfilScoring',
            'FechaRecepcionDOC',
            'FechaFacturacion',
            'RepercusionCosto',
            'Desplazamiento',
            'ResumenVenta',
            'ProspectoAbierto',
            'FechaProspectoAbierto',
            'pagoAdelantado',
            'conSistema',
            'empresaSistema',
            'vueltasTeleventa',
            'ultimoGestorTeleventa',
            'ResultadoGestionTeleventa',
            'SubtipificacionTeleventa',
            'MotivoSubtipificacionTeleventa',
            'FechaHoraTransferenciaTCTeleventa',
            'vueltasTCTeleventa',
            'ultimoGestorTCTeleventa',
            'ResultadoGestionTCTeleventa',
            'SubtipificacionTCTeleventa',
            'MotivoSubtipificacionTCTeleventa'
            ];
        $this->reportWriter->writeHeaderRow($headerRow);
    }
    
   private function fetchRowsOneByOneAndWriteThem_TCTeleventa($query){
        $dbRowIterator = new SingleFetchIterator($this->pdo, $query);

        foreach ($dbRowIterator as $dbRow) {

        $reportRow = [
            $dbRow['IdVenta'],
            $dbRow['IdCliente'],
            $dbRow['Telefonouno'],
            $dbRow['Telefonodos'],
            $dbRow['Cliente'],
            $dbRow['EstadoTeleventa'],
            $dbRow['EstadoVenta'],
            $dbRow['TipoVenta'],
            $dbRow['FechaHoraREG'],
            $dbRow['TipoTV'],
            $dbRow['TCGestor'],
            $dbRow['NombreTV'],
            $dbRow['Segmento'],
            $dbRow['PrecioKitTV'],
            $dbRow['N590'],
            $dbRow['Departamento'],
            $dbRow['Provincia'],
            $dbRow['Distrito'],
            $dbRow['Zonas'],
            $dbRow['Vueltas'],
            $dbRow['OrigenVisita'],
            $dbRow['Canal'],
            $dbRow['FechaInst'],
            $dbRow['HoraInst'],
            $dbRow['CC'],
            $dbRow['MatriculaTVOriginal'],
            $dbRow['NProspecto'],
            $dbRow['NContrato'],
            $dbRow['IdLead'],
            $dbRow['TipoTransferencia'],
            $dbRow['FechaRellamadaTV'],
            $dbRow['HoraRellamadaTV'],
            $dbRow['Contacto'],
            $dbRow['Gestionable'],
            $dbRow['SubEstadoTeleventa'],
            $dbRow['SubCompetencia'],
            $dbRow['ultimoGestorCC'],
            $dbRow['ultimoNombreGestorCC'],
            $dbRow['ObsTransferencia'],
            $dbRow['FechaUltimaGestion'],
            $dbRow['FechaUltimaGestionCC'],
            $dbRow['LeadID'],
            $dbRow['Repercusion'],
            $dbRow['ComentarioAgenda'],
            $dbRow['TipoKIT'],
            $dbRow['TotalExtra'],
            $dbRow['CantidadExtrasTV'],
            $dbRow['ComentarioTV'],
            $dbRow['ComentarioCancelacion'],
            $dbRow['TipoReferido'],
            $dbRow['referidoHabilitadoTVTC'],
            $dbRow['TotalCuotaMonitoreo'],
            $dbRow['CantidadRegalo'],
            $dbRow['DescMeses'],
            $dbRow['PAI'],
            $dbRow['TCScoring'],
            $dbRow['PerfilScoring'],
            $dbRow['FechaRecepcionDOC'],
            $dbRow['FechaFacturacion'],
            $dbRow['RepercusionCosto'],
            $dbRow['Desplazamiento'],
            $dbRow['ResumenVenta'],
            $dbRow['ProspectoAbierto'],
            $dbRow['FechaProspectoAbierto'],
            $dbRow['pagoAdelantado'],
            $dbRow['conSistema'],
            $dbRow['empresaSistema'],
            $dbRow['vueltasTeleventa'],
            $dbRow['ultimoGestorTeleventa'],
            $dbRow['ResultadoGestionTeleventa'],
            $dbRow['SubtipificacionTeleventa'],
            $dbRow['MotivoSubtipificacionTeleventa'],
            $dbRow['FechaHoraTransferenciaTCTeleventa'],
            $dbRow['vueltasTCTeleventa'],
            $dbRow['ultimoGestorTCTeleventa'],
            $dbRow['ResultadoGestionTCTeleventa'],
            $dbRow['SubtipificacionTCTeleventa'],
            $dbRow['MotivoSubtipificacionTCTeleventa']
                    ];
            $this->reportWriter->writeRow($reportRow);
        }
    }     

  //FIN exportTCTeleventa  
    
}
