<?php

namespace SpoutExample;

use SpoutExample\Iterator\BatchFetchIterator;
use SpoutExample\Iterator\SingleFetchIterator;
use SpoutExample\ReportWriter\PHPExcelWriter;
use SpoutExample\ReportWriter\SpoutWriter;
use SpoutExample\ReportWriter\WriterType;

/**
 * Class ReportCreatorAgenda
 * In this example, we want to generate a report listing all the products available
 * as well as some associated data (quantity available, quantity sold, ...).
 *
 * @package SpoutExample
 */
class ReportCreatorAttec
{
    /** Fetching methods */
    const FETCH_ROWS_ONE_BY_ONE = 1;
    const FETCH_ROWS_IN_BATCH = 2;
    const FETCH_ROWS_ALL_AT_ONCE = 3;

    /** @var \PDO PDO instance */
    private $pdo;

    /** @var ReportWriter\WriterType Type of writer to use */
    private $writerType = WriterType::SPOUT;

    /** @var ReportWriter\AbstractWriter Writer used to create a XLSX report */
    private $reportWriter;

    /** @var int The fetching method to use, used for benchmarks */
    private $fetchingMethod = self::FETCH_ROWS_IN_BATCH;

    /** @var int Number of rows to fetch for each batch (used only when FETCH_ROWS_IN_BATCH is enabled) */
    private $batchSize;

    /**
     * @param DBConf $dbConf
     */
    public function __construct(DBConf $dbConf)
    {
        $this->pdo = new \PDO($dbConf->getDSN(), $dbConf->getUsername(), $dbConf->getPassword());
    }

    /**
     * @return self
     */
    public function setFetchRowsOneByOne()
    {
        $this->fetchingMethod = self::FETCH_ROWS_ONE_BY_ONE;
        return $this;
    }

    /**
     * @param int $batchSize Number of rows to fetch for each batch
     * @return self
     */
    public function setFetchRowsInBatch($batchSize)
    {
        $this->fetchingMethod = self::FETCH_ROWS_IN_BATCH;
        $this->batchSize = $batchSize;
        return $this;
    }

    /**
     * @return self
     */
    public function setFetchAllRowsAtOnce()
    {
        $this->fetchingMethod = self::FETCH_ROWS_ALL_AT_ONCE;
        return $this;
    }

    /**
     * @return string Name of the fetching method used
     */
    public function getFetchMethodName()
    {
        switch ($this->fetchingMethod) {
            case self::FETCH_ROWS_ONE_BY_ONE: return 'Fetch mode: one by one';
            case self::FETCH_ROWS_ALL_AT_ONCE: return 'Fetch mode: all at once';
            case self::FETCH_ROWS_IN_BATCH:
            default:
                return 'Fetch mode: batch';
        }
    }

    /**
     * @see \SpoutExample\ReportWriter\WriterType
     *
     * @param string $writerType
     * @return ReportCreator
     */
    public function setWriterType($writerType)
    {
        $this->writerType = $writerType;
        return $this;
    }

    /**
     * @param $outputPath
     * @return ReportWriter\AbstractWriter
     */
    private function createReportWriter($outputPath)
    {
        switch ($this->writerType) {
            case WriterType::PHP_EXCEL: return new PHPExcelWriter($outputPath);
            case WriterType::SPOUT:
            default:
                return new SpoutWriter($outputPath);
        }
    }

    /**
     * Fetches the data from the DB and spits it out in a XLSX file.
     *
     * @param string $outputPath Path where the report will be written to
     * @return void
     */
    public function fetchDataAndCreateReport($outputPath)
    {
        $this->reportWriter = $this->createReportWriter($outputPath);
        $this->writeReportHeader();

        // Make sure to only select the fields we are interested in
        $query = "SELECT 
v.IdVenta,
v.IdUsuario,
v.IdCliente,
v.Matricula,
v.Comercial,
v.CC,
v.JefeVenta,
v.CategoriaVenta,
v.TipoVenta,
v.Segmento,
v.NProspecto,
v.NContrato,
v.PrecioKit,
v.TotalExtra,
v.IngresoKIT,
v.IngresoExtras,
v.IngresoTotal,
v.EstadoVenta,
v.ExtraAutorizado,
v.MesesAutorizado,
v.N590,
v.FormaPago,
v.PAI,
v.ResumenVenta,
v.FechaHoraREG,
v.EstadoVisita,
v.FechaVisita,
v.HoraVisita,
v.MesVisita,
v.FVencimientoVisita,
v.GestorMKT,
v.OrigenVisita,
v.ComentarioVisita,
v.GestorOrigen,
c.IdGeografico,
c.NombApellido,
c.Telefonouno,
c.Telefonodos,
c.Correo,
c.Direccion,
g.Distrito,
g.Provincia,
v.CCOriginal,
v.EstadoTelecierre,
v.TCGestor,
v.TCMotivoNoVenta,
v.TCTipoVenta,
v.Checkin,
v.LeadID,
concat(a.FechaInst, ' ', a.HoraInst) FechaHoraInst,
d.TipoDocumento,
d.Documento,
v.Accion,
v.PerfilScoring,
v.IdConsultaScoring,
s.Distrital as DistritalScoring,
s.Provincial as ProvincialScoring,
s.Departamental as DepartamentalScoring,
s.GestorOrigen as MatriculaScoring,
s.Score,
s.PredictorIngresos_IP,
s.CantidadDocumentosMorosos,
s.FechaHoraReg as FechaHoraConsulta,
v.IMP,
c.objGeofinder
FROM yifi58ge_dbVENTAS.tblventa v INNER JOIN yifi58ge_dbVENTAS.tblcliente c on c.IdCliente=v.IdCliente
INNER JOIN yifi58ge_dbVENTAS.tblagenda a ON a.IdVenta=v.IdVenta 
INNER JOIN yifi58ge_dbVENTAS.tblgeografico g ON g.IdGeografico=c.IdGeografico
LEFT JOIN yifi58ge_dbVENTAS.tbldocumento d ON c.IdDocVisita = d.IdDoc
LEFT JOIN yifi58ge_dbVENTAS.tbltmpscoring s ON s.IdConsultaScoring=v.IdConsultaScoring    
WHERE v.TipoVenta='RE'
and v.IdVenta >= 45451";

        switch ($this->fetchingMethod) {
            case self::FETCH_ROWS_ONE_BY_ONE:
                $this->fetchRowsOneByOneAndWriteThem($query);
                break;
            case self::FETCH_ROWS_IN_BATCH:
                $this->fetchRowsInBatchAndWriteThem($query);
                break;
            case self::FETCH_ROWS_ALL_AT_ONCE:
                $this->fetchAllRowsAtOnceAndWriteThem($query);
                break;
        }

        $this->reportWriter->close();
    }

    /**
     * @param string $query
     * @param int $maxFetchedRows
     * @return void
     */
    private function fetchRowsOneByOneAndWriteThem($query)
    {
        $dbRowIterator = new SingleFetchIterator($this->pdo, $query);

        foreach ($dbRowIterator as $dbRow) {
                if(!empty($dbRow['objGeofinder']) || !is_null($dbRow['objGeofinder'])){
                    $objGeofinder = json_decode($dbRow['objGeofinder'], true);
                    //$InfoCheckIn = json_decode($Venta['InfoCheckIn'], true);
                    $zonas = $objGeofinder['zonas'];
                    //ZONA
                    if(!empty($zonas[0]['valor'])){
                       $zona = $zonas[0]['valor']; 
                    }else{
                      $zona = '';  
                    }
                    //MICROZONA
                    if(!empty($zonas[1]['valor'])){
                       //$microzona = $zonas[1]['valor']; 
                       
                       $microzona = convertirMicrozona($zonas[1]['valor']);
                       
                    }else{
                      $microzona = '';  
                    }                    
                }else{
                    //ZONA
                    $zona = '';
                    //MICROZONA
                    $microzona='';
                }
            
            
            $reportRow = [$dbRow['IdVenta'],
$dbRow['IdUsuario'],
$dbRow['IdCliente'],
$dbRow['Matricula'],
$dbRow['Comercial'],
$dbRow['CC'],
$dbRow['JefeVenta'],
$dbRow['CategoriaVenta'],
$dbRow['TipoVenta'],
$dbRow['Segmento'],
$dbRow['NProspecto'],
$dbRow['NContrato'],
$dbRow['PrecioKit'],
$dbRow['TotalExtra'],
$dbRow['IngresoKIT'],
$dbRow['IngresoExtras'],
$dbRow['IngresoTotal'],
$dbRow['EstadoVenta'],
$dbRow['ExtraAutorizado'],
$dbRow['MesesAutorizado'],
$dbRow['N590'],
$dbRow['FormaPago'],
$dbRow['PAI'],
$dbRow['ResumenVenta'],
$dbRow['FechaHoraREG'],
$dbRow['EstadoVisita'],
$dbRow['FechaVisita'],
$dbRow['HoraVisita'],
$dbRow['MesVisita'],
$dbRow['FVencimientoVisita'],
$dbRow['GestorMKT'],
$dbRow['OrigenVisita'],
$dbRow['ComentarioVisita'],
$dbRow['GestorOrigen'],
$dbRow['IdGeografico'],
$dbRow['NombApellido'],
$dbRow['Telefonouno'],
$dbRow['Telefonodos'],
$dbRow['Correo'],
$dbRow['Direccion'],
$dbRow['Distrito'],
$dbRow['Provincia'],
$dbRow['CCOriginal'],
$dbRow['EstadoTelecierre'],
$dbRow['TCGestor'],
$dbRow['TCMotivoNoVenta'],
$dbRow['TCTipoVenta'],
$dbRow['Checkin'],
$dbRow['LeadID'],
$dbRow['FechaHoraInst'],
$dbRow['TipoDocumento'],
$dbRow['Documento'],
$dbRow['Accion'],
$dbRow['PerfilScoring'],
$dbRow['IdConsultaScoring'],
$dbRow['DistritalScoring'],
$dbRow['ProvincialScoring'],
$dbRow['DepartamentalScoring'],
$dbRow['MatriculaScoring'],
$dbRow['Score'],
$dbRow['PredictorIngresos_IP'],
$dbRow['CantidadDocumentosMorosos'],
$dbRow['FechaHoraConsulta'],
$zona,
$microzona,
$dbRow['IMP']
];
            $this->reportWriter->writeRow($reportRow);
        }
    }

    /**
     * @param string $query
     * @param int $maxFetchedRows
     * @return void
     */
    private function fetchRowsInBatchAndWriteThem($query)
    {
        $idFieldName = 'IdVenta';
        $batchFetchIterator = new BatchFetchIterator($this->pdo, $query, $idFieldName, $this->batchSize);

        foreach ($batchFetchIterator as $dbRows) {
            foreach ($dbRows as $dbRow) {
                $reportRow = [$dbRow['IdVenta'], $dbRow['IdUsuario'], $dbRow['IdCliente'], $dbRow['Matricula'], $dbRow['Comercial'], $dbRow['CC']];
                $this->reportWriter->writeRow($reportRow);
            }
        }
    }

    /**
     * @param string $query
     * @return void
     */
    private function fetchAllRowsAtOnceAndWriteThem($query)
    {
        $statement = $this->pdo->prepare($query);
        $statement->execute();

        $allDBRows = $statement->fetchAll(\PDO::FETCH_ASSOC);

        foreach ($allDBRows as $dbRow) {
            $reportRow =  [$dbRow['IdVenta'], $dbRow['IdUsuario'], $dbRow['IdCliente'], $dbRow['Matricula'], $dbRow['Comercial'], $dbRow['CC']];
            $this->reportWriter->writeRow($reportRow);
        }

        $statement->closeCursor();
    }

    /**
     * @return void
     */
    private function writeReportHeader()
    {
        // The header will be bold
        $headerRow = ['IdVenta',
'IdUsuario',
'IdCliente',
'Matricula',
'Comercial',
'CC',
'JefeVenta',
'CategoriaVenta',
'TipoVenta',
'Segmento',
'NProspecto',
'NContrato',
'PrecioKit',
'TotalExtra',
'IngresoKIT',
'IngresoExtras',
'IngresoTotal',
'EstadoVenta',
'ExtraAutorizado',
'MesesAutorizado',
'N590',
'FormaPago',
'PAI',
'ResumenVenta',
'FechaHoraREG',
'EstadoVisita',
'FechaVisita',
'HoraVisita',
'MesVisita',
'FVencimientoVisita',
'GestorMKT',
'OrigenVisita',
'ComentarioVisita',
'GestorOrigen',
'IdGeografico',
'NombApellido',
'Telefonouno',
'Telefonodos',
'Correo',
'Direccion',
'Distrito',
'Provincia',
'CCOriginal',
'EstadoTelecierre',
'TCGestor',
'TCMotivoNoVenta',
'TCTipoVenta',
'Checkin',
'LeadID',
'FechaHoraInst',
'TipoDocumento',
'Documento',
'Accion',
'PerfilScoring',
'IdConsultaScoring',
'DistritalScoring',
'ProvincialScoring',
'DepartamentalScoring',
'MatriculaScoring',
'Score',
'PredictorIngresos_IP',
'CantidadDocumentosMorosos',
'FechaHoraConsulta',
'Zona',
'Seccion',
'IMP'];
        $this->reportWriter->writeHeaderRow($headerRow);
    }
  
    //FUNCIONES DE DESCARGA DE AttecAuditoria
    public function fetchDataAndCreateReport_AttecAuditoria($outputPath){
        $this->reportWriter = $this->createReportWriter($outputPath);
        $this->writeReportHeader_AttecAuditoria();
        // Make sure to only select the fields we are interested in
        $query="SELECT
        v.IdVenta,
        u.IdUsuario,
        v.Matricula,
        v.Comercial,
        v.CC,
        v.JefeVenta,
        CONCAT(m.Nombres,' ',m.Apellidos) NombJefeVenta,
        v.EstadoVenta,
        v.CategoriaVenta,
        v.TipoVenta,
        v.Segmento,
        v.MesesAutorizado,
        v.NProspecto,
        v.NContrato,
        v.IdCliente,
        c.NombApellido Cliente,
        c.Correo,
        c.Telefonouno,
        c.Telefonodos,
        c.Direccion,
        c.Referencia,
        c.Interior,
        g.IdGeografico,
        g.Distrito,
        g.Provincia,
        v.PrecioKit,
        v.TotalExtra,
        v.ResumenVenta,
        v.ExtraAutorizado,
        v.FormaPago,
        v.N590,
        v.PAI,
        v.CCOriginal,
        agen.IdAgenda,
        agen.InstaladoX,
        agen.MatriculaTec,
        CONCAT(t.Nombres,' ',t.Apellidos) Instalador,
        t.LN LNInstalador,
        agen.FechaInst,
        agen.HoraInst,
        agen.Comentario,
        agen.MotivoInviable,
        agen.MotivoCancelada,
        agen.Repercusion,
        agen.RepercusionCosto,
        v.TipoKIT,
        att.IdAttec,
        att.FechaAuditoria,
        att.EstadoAttec,
        att.DatosSbn,
        att.F_CorreccionDatosSbn,
        att.SegIncorrecto,
        att.F_CorreccionSegIncorrecto,
        att.Seg_DebeDecir,
        att.Imagen,
        att.F_Imagen,
        att.Imagen_Motivo,
        att.IngresoProgramacion,
        att.F_CorreccionIngresoProgramacion,
        att.FallosCobertura,
        att.F_CorreccionFallosCobertura,
        att.MissingTest,
        att.F_CorreccionMissingTest,
        att.ZeroVision,
        att.F_CorreccionZeroVision,
        att.FalloSupervision,
        att.F_CorreccionFalloSupervision,
        att.FalloSupervision_Dispotivos,
        att.Tamper,
        att.F_CorreccionTamper,
        att.Tamper_Dispotivos,
        att.MG_01_retardo,
        att.F_CorreccionMG_01_retardo,
        f.FechaFacturacion
        FROM yifi58ge_dbVENTAS.tblventa v 
        INNER JOIN yifi58ge_dbUSUARIO.tblusuario u on v.IdUsuario=u.IdUsuario 
        INNER JOIN yifi58ge_dbUSUARIO.tblarea a on a.IdArea=u.IdArea 
        INNER JOIN yifi58ge_dbUSUARIO.tblusuario m on m.Matricula=u.Manager
        INNER JOIN yifi58ge_dbVENTAS.tblcliente c on c.IdCliente=v.IdCliente
        INNER JOIN yifi58ge_dbVENTAS.tblgeografico g on g.IdGeografico=c.IdGeografico
        INNER JOIN yifi58ge_dbVENTAS.tblagenda agen on agen.IdVenta=v.IdVenta
        INNER JOIN yifi58ge_dbVENTAS.tblattec att on att.IdVenta=v.IdVenta
        INNER JOIN yifi58ge_dbVENTAS.tblfacturacion f on f.IdVenta=v.IdVenta
        LEFT JOIN yifi58ge_dbUSUARIO.tblusuario t on t.Matricula=agen.MatriculaTec        
        ";
        switch ($this->fetchingMethod) {
            case self::FETCH_ROWS_ONE_BY_ONE:
                $this->fetchRowsOneByOneAndWriteThem_AttecAuditoria($query);
                break;
            case self::FETCH_ROWS_IN_BATCH:
                $this->fetchRowsInBatchAndWriteThem($query);
                break;
            case self::FETCH_ROWS_ALL_AT_ONCE:
                $this->fetchAllRowsAtOnceAndWriteThem($query);
                break;
        }

        $this->reportWriter->close();
    }
    private function writeReportHeader_AttecAuditoria(){
        // The header will be bold
        $headerRow = [
            'IdVenta',
            'IdUsuario',
            'Matricula',
            'Comercial',
            'CC',
            'JefeVenta',
            'NombJefeVenta',
            'EstadoVenta',
            'CategoriaVenta',
            'TipoVenta',
            'Segmento',
            'MesesAutorizado',
            'NProspecto',
            'NContrato',
            'IdCliente',
            'Cliente',
            'Correo',
            'Telefonouno',
            'Telefonodos',
            'Direccion',
            'Referencia',
            'Interior',
            'IdGeografico',
            'Distrito',
            'Provincia',
            'PrecioKit',
            'TotalExtra',
            'ResumenVenta',
            'ExtraAutorizado',
            'FormaPago',
            'N590',
            'PAI',
            'CCOriginal',
            'IdAgenda',
            'InstaladoX',
            'MatriculaTec',
            'Instalador',
            'LNInstalador',
            'FechaInst',
            'HoraInst',
            'Comentario',
            'MotivoInviable',
            'MotivoCancelada',
            'Repercusion',
            'RepercusionCosto',
            'TipoKIT',
            'IdAttec',
            'FechaAuditoria',
            'EstadoAttec',
            'DatosSbn',
            'F_CorreccionDatosSbn',
            'SegIncorrecto',
            'F_CorreccionSegIncorrecto',
            'Seg_DebeDecir',
            'Imagen',
            'F_Imagen',
            'Imagen_Motivo',
            'IngresoProgramacion',
            'F_CorreccionIngresoProgramacion',
            'FallosCobertura',
            'F_CorreccionFallosCobertura',
            'MissingTest',
            'F_CorreccionMissingTest',
            'ZeroVision',
            'F_CorreccionZeroVision',
            'FalloSupervision',
            'F_CorreccionFalloSupervision',
            'FalloSupervision_Dispotivos',
            'Tamper',
            'F_CorreccionTamper',
            'Tamper_Dispotivos',
            'MG_01_retardo',
            'F_CorreccionMG_01_retardo',            
            'FechaFacturacion'
            ];
        $this->reportWriter->writeHeaderRow($headerRow);
    }
    private function fetchRowsOneByOneAndWriteThem_AttecAuditoria($query){
        $dbRowIterator = new SingleFetchIterator($this->pdo, $query);
        foreach ($dbRowIterator as $dbRow) {
        $reportRow = [
            $dbRow['IdVenta'],
            $dbRow['IdUsuario'],
            $dbRow['Matricula'],
            $dbRow['Comercial'],
            $dbRow['CC'],
            $dbRow['JefeVenta'],
            $dbRow['NombJefeVenta'],
            $dbRow['EstadoVenta'],
            $dbRow['CategoriaVenta'],
            $dbRow['TipoVenta'],
            $dbRow['Segmento'],
            $dbRow['MesesAutorizado'],
            $dbRow['NProspecto'],
            $dbRow['NContrato'],
            $dbRow['IdCliente'],
            $dbRow['Cliente'],
            $dbRow['Correo'],
            $dbRow['Telefonouno'],
            $dbRow['Telefonodos'],
            $dbRow['Direccion'],
            $dbRow['Referencia'],
            $dbRow['Interior'],
            $dbRow['IdGeografico'],
            $dbRow['Distrito'],
            $dbRow['Provincia'],
            $dbRow['PrecioKit'],
            $dbRow['TotalExtra'],
            $dbRow['ResumenVenta'],
            $dbRow['ExtraAutorizado'],
            $dbRow['FormaPago'],
            $dbRow['N590'],
            $dbRow['PAI'],
            $dbRow['CCOriginal'],
            $dbRow['IdAgenda'],
            $dbRow['InstaladoX'],
            $dbRow['MatriculaTec'],
            $dbRow['Instalador'],
            $dbRow['LNInstalador'],
            $dbRow['FechaInst'],
            $dbRow['HoraInst'],
            $dbRow['Comentario'],
            $dbRow['MotivoInviable'],
            $dbRow['MotivoCancelada'],
            $dbRow['Repercusion'],
            $dbRow['RepercusionCosto'],
            $dbRow['TipoKIT'],
            $dbRow['IdAttec'],
            $dbRow['FechaAuditoria'],
            $dbRow['EstadoAttec'],
            $dbRow['DatosSbn'],
            $dbRow['F_CorreccionDatosSbn'],
            $dbRow['SegIncorrecto'],
            $dbRow['F_CorreccionSegIncorrecto'],
            $dbRow['Seg_DebeDecir'],
            $dbRow['Imagen'],
            $dbRow['F_Imagen'],
            $dbRow['Imagen_Motivo'],
            $dbRow['IngresoProgramacion'],
            $dbRow['F_CorreccionIngresoProgramacion'],
            $dbRow['FallosCobertura'],
            $dbRow['F_CorreccionFallosCobertura'],
            $dbRow['MissingTest'],
            $dbRow['F_CorreccionMissingTest'],
            $dbRow['ZeroVision'],
            $dbRow['F_CorreccionZeroVision'],
            $dbRow['FalloSupervision'],
            $dbRow['F_CorreccionFalloSupervision'],
            $dbRow['FalloSupervision_Dispotivos'],
            $dbRow['Tamper'],
            $dbRow['F_CorreccionTamper'],
            $dbRow['Tamper_Dispotivos'],
            $dbRow['MG_01_retardo'],
            $dbRow['F_CorreccionMG_01_retardo'],
            $dbRow['FechaFacturacion']
            ];
            $this->reportWriter->writeRow($reportRow);
        }
    }       

  //FIN AttecAuditoria


    
}
