<?php

namespace SpoutExample;

use SpoutExample\Iterator\BatchFetchIterator;
use SpoutExample\Iterator\SingleFetchIterator;
use SpoutExample\ReportWriter\PHPExcelWriter;
use SpoutExample\ReportWriter\SpoutWriter;
use SpoutExample\ReportWriter\WriterType;

/**
 * Class ReportCreator
 * In this example, we want to generate a report listing all the products available
 * as well as some associated data (quantity available, quantity sold, ...).
 *
 * @package SpoutExample
 */
class ReportCreator
{
    /** Fetching methods */
    const FETCH_ROWS_ONE_BY_ONE = 1;
    const FETCH_ROWS_IN_BATCH = 2;
    const FETCH_ROWS_ALL_AT_ONCE = 3;

    /** @var \PDO PDO instance */
    private $pdo;

    /** @var ReportWriter\WriterType Type of writer to use */
    private $writerType = WriterType::SPOUT;

    /** @var ReportWriter\AbstractWriter Writer used to create a XLSX report */
    private $reportWriter;

    /** @var int The fetching method to use, used for benchmarks */
    private $fetchingMethod = self::FETCH_ROWS_IN_BATCH;

    /** @var int Number of rows to fetch for each batch (used only when FETCH_ROWS_IN_BATCH is enabled) */
    private $batchSize;

    /**
     * @param DBConf $dbConf
     */
    public function __construct(DBConf $dbConf)
    {
        $this->pdo = new \PDO($dbConf->getDSN(), $dbConf->getUsername(), $dbConf->getPassword());
    }

    /**
     * @return self
     */
    public function setFetchRowsOneByOne()
    {
        $this->fetchingMethod = self::FETCH_ROWS_ONE_BY_ONE;
        return $this;
    }

    /**
     * @param int $batchSize Number of rows to fetch for each batch
     * @return self
     */
    public function setFetchRowsInBatch($batchSize)
    {
        $this->fetchingMethod = self::FETCH_ROWS_IN_BATCH;
        $this->batchSize = $batchSize;
        return $this;
    }

    /**
     * @return self
     */
    public function setFetchAllRowsAtOnce()
    {
        $this->fetchingMethod = self::FETCH_ROWS_ALL_AT_ONCE;
        return $this;
    }

    /**
     * @return string Name of the fetching method used
     */
    public function getFetchMethodName()
    {
        switch ($this->fetchingMethod) {
            case self::FETCH_ROWS_ONE_BY_ONE: return 'Fetch mode: one by one';
            case self::FETCH_ROWS_ALL_AT_ONCE: return 'Fetch mode: all at once';
            case self::FETCH_ROWS_IN_BATCH:
            default:
                return 'Fetch mode: batch';
        }
    }

    /**
     * @see \SpoutExample\ReportWriter\WriterType
     *
     * @param string $writerType
     * @return ReportCreator
     */
    public function setWriterType($writerType)
    {
        $this->writerType = $writerType;
        return $this;
    }

    /**
     * @param $outputPath
     * @return ReportWriter\AbstractWriter
     */
    private function createReportWriter($outputPath)
    {
        switch ($this->writerType) {
            case WriterType::PHP_EXCEL: return new PHPExcelWriter($outputPath);
            case WriterType::SPOUT:
            default:
                return new SpoutWriter($outputPath);
        }
    }

    /**
     * Fetches the data from the DB and spits it out in a XLSX file.
     *
     * @param string $outputPath Path where the report will be written to
     * @return void
     */
    public function fetchDataAndCreateReport($outputPath)
    {
        $this->reportWriter = $this->createReportWriter($outputPath);
        $this->writeReportHeader();

        // Make sure to only select the fields we are interested in
        $query = "SELECT 
v.IdVenta,
v.IdUsuario,
v.IdCliente,
v.Matricula,
v.Comercial,
v.CC,
v.JefeVenta,
v.CategoriaVenta,
v.TipoVenta,
v.Segmento,
v.NProspecto,
v.NContrato,
v.PrecioKit,
v.TotalExtra,
v.IngresoKIT,
v.IngresoExtras,
v.IngresoTotal,
v.EstadoVenta,
v.ExtraAutorizado,
v.MesesAutorizado,
v.N590,
v.FormaPago,
v.PAI,
v.ResumenVenta,
v.FechaHoraREG,
v.EstadoVisita,
v.FechaVisita,
v.HoraVisita,
v.MesVisita,
v.FVencimientoVisita,
v.GestorMKT,
v.OrigenVisita,
v.ComentarioVisita,
v.GestorOrigen,
c.IdGeografico,
c.NombApellido,
c.Telefonouno,
c.Telefonodos,
c.Correo,
c.Direccion,
g.Distrito,
g.Provincia,
v.CCOriginal,
v.EstadoTelecierre,
v.TCGestor,
v.TCMotivoNoVenta,
v.TCTipoVenta,
v.Checkin,
v.LeadID,
concat(a.FechaInst, ' ', a.HoraInst) FechaHoraInst,
d.TipoDocumento,
d.Documento,
v.Accion,
v.PerfilScoring,
v.IdConsultaScoring,
s.Distrital as DistritalScoring,
s.Provincial as ProvincialScoring,
s.Departamental as DepartamentalScoring,
s.GestorOrigen as MatriculaScoring,
s.Score,
s.PredictorIngresos_IP,
s.CantidadDocumentosMorosos,
s.FechaHoraReg as FechaHoraConsulta,
v.IMP,
c.objGeofinder
FROM yifi58ge_dbVENTAS.tblventa v INNER JOIN yifi58ge_dbVENTAS.tblcliente c on c.IdCliente=v.IdCliente
INNER JOIN yifi58ge_dbVENTAS.tblagenda a ON a.IdVenta=v.IdVenta 
INNER JOIN yifi58ge_dbVENTAS.tblgeografico g ON g.IdGeografico=c.IdGeografico
LEFT JOIN yifi58ge_dbVENTAS.tbldocumento d ON c.IdDocVisita = d.IdDoc
LEFT JOIN yifi58ge_dbVENTAS.tbltmpscoring s ON s.IdConsultaScoring=v.IdConsultaScoring    
WHERE v.TipoVenta='RE'
and v.IdVenta >= 45451";

        switch ($this->fetchingMethod) {
            case self::FETCH_ROWS_ONE_BY_ONE:
                $this->fetchRowsOneByOneAndWriteThem($query);
                break;
            case self::FETCH_ROWS_IN_BATCH:
                $this->fetchRowsInBatchAndWriteThem($query);
                break;
            case self::FETCH_ROWS_ALL_AT_ONCE:
                $this->fetchAllRowsAtOnceAndWriteThem($query);
                break;
        }

        $this->reportWriter->close();
    }

    /**
     * @param string $query
     * @param int $maxFetchedRows
     * @return void
     */
    private function fetchRowsOneByOneAndWriteThem($query)
    {
        $dbRowIterator = new SingleFetchIterator($this->pdo, $query);

        foreach ($dbRowIterator as $dbRow) {
                if(!empty($dbRow['objGeofinder']) || !is_null($dbRow['objGeofinder'])){
                    $objGeofinder = json_decode($dbRow['objGeofinder'], true);
                    //$InfoCheckIn = json_decode($Venta['InfoCheckIn'], true);
                    $zonas = $objGeofinder['zonas'];
                    //ZONA
                    if(!empty($zonas[0]['valor'])){
                       $zona = $zonas[0]['valor']; 
                    }else{
                      $zona = '';  
                    }
                    //MICROZONA
                    if(!empty($zonas[1]['valor'])){
                       //$microzona = $zonas[1]['valor']; 
                       
                       $microzona = convertirMicrozona($zonas[1]['valor']);
                       
                    }else{
                      $microzona = '';  
                    }                    
                }else{
                    //ZONA
                    $zona = '';
                    //MICROZONA
                    $microzona='';
                }
            
            
            $reportRow = [$dbRow['IdVenta'],
$dbRow['IdUsuario'],
$dbRow['IdCliente'],
$dbRow['Matricula'],
$dbRow['Comercial'],
$dbRow['CC'],
$dbRow['JefeVenta'],
$dbRow['CategoriaVenta'],
$dbRow['TipoVenta'],
$dbRow['Segmento'],
$dbRow['NProspecto'],
$dbRow['NContrato'],
$dbRow['PrecioKit'],
$dbRow['TotalExtra'],
$dbRow['IngresoKIT'],
$dbRow['IngresoExtras'],
$dbRow['IngresoTotal'],
$dbRow['EstadoVenta'],
$dbRow['ExtraAutorizado'],
$dbRow['MesesAutorizado'],
$dbRow['N590'],
$dbRow['FormaPago'],
$dbRow['PAI'],
$dbRow['ResumenVenta'],
$dbRow['FechaHoraREG'],
$dbRow['EstadoVisita'],
$dbRow['FechaVisita'],
$dbRow['HoraVisita'],
$dbRow['MesVisita'],
$dbRow['FVencimientoVisita'],
$dbRow['GestorMKT'],
$dbRow['OrigenVisita'],
$dbRow['ComentarioVisita'],
$dbRow['GestorOrigen'],
$dbRow['IdGeografico'],
$dbRow['NombApellido'],
$dbRow['Telefonouno'],
$dbRow['Telefonodos'],
$dbRow['Correo'],
$dbRow['Direccion'],
$dbRow['Distrito'],
$dbRow['Provincia'],
$dbRow['CCOriginal'],
$dbRow['EstadoTelecierre'],
$dbRow['TCGestor'],
$dbRow['TCMotivoNoVenta'],
$dbRow['TCTipoVenta'],
$dbRow['Checkin'],
$dbRow['LeadID'],
$dbRow['FechaHoraInst'],
$dbRow['TipoDocumento'],
$dbRow['Documento'],
$dbRow['Accion'],
$dbRow['PerfilScoring'],
$dbRow['IdConsultaScoring'],
$dbRow['DistritalScoring'],
$dbRow['ProvincialScoring'],
$dbRow['DepartamentalScoring'],
$dbRow['MatriculaScoring'],
$dbRow['Score'],
$dbRow['PredictorIngresos_IP'],
$dbRow['CantidadDocumentosMorosos'],
$dbRow['FechaHoraConsulta'],
$zona,
$microzona,
$dbRow['IMP']
];
            $this->reportWriter->writeRow($reportRow);
        }
    }

    /**
     * @param string $query
     * @param int $maxFetchedRows
     * @return void
     */
    private function fetchRowsInBatchAndWriteThem($query)
    {
        $idFieldName = 'IdVenta';
        $batchFetchIterator = new BatchFetchIterator($this->pdo, $query, $idFieldName, $this->batchSize);

        foreach ($batchFetchIterator as $dbRows) {
            foreach ($dbRows as $dbRow) {
                $reportRow = [$dbRow['IdVenta'], $dbRow['IdUsuario'], $dbRow['IdCliente'], $dbRow['Matricula'], $dbRow['Comercial'], $dbRow['CC']];
                $this->reportWriter->writeRow($reportRow);
            }
        }
    }

    /**
     * @param string $query
     * @return void
     */
    private function fetchAllRowsAtOnceAndWriteThem($query)
    {
        $statement = $this->pdo->prepare($query);
        $statement->execute();

        $allDBRows = $statement->fetchAll(\PDO::FETCH_ASSOC);

        foreach ($allDBRows as $dbRow) {
            $reportRow =  [$dbRow['IdVenta'], $dbRow['IdUsuario'], $dbRow['IdCliente'], $dbRow['Matricula'], $dbRow['Comercial'], $dbRow['CC']];
            $this->reportWriter->writeRow($reportRow);
        }

        $statement->closeCursor();
    }

    /**
     * @return void
     */
    private function writeReportHeader()
    {
        // The header will be bold
        $headerRow = ['IdVenta',
'IdUsuario',
'IdCliente',
'Matricula',
'Comercial',
'CC',
'JefeVenta',
'CategoriaVenta',
'TipoVenta',
'Segmento',
'NProspecto',
'NContrato',
'PrecioKit',
'TotalExtra',
'IngresoKIT',
'IngresoExtras',
'IngresoTotal',
'EstadoVenta',
'ExtraAutorizado',
'MesesAutorizado',
'N590',
'FormaPago',
'PAI',
'ResumenVenta',
'FechaHoraREG',
'EstadoVisita',
'FechaVisita',
'HoraVisita',
'MesVisita',
'FVencimientoVisita',
'GestorMKT',
'OrigenVisita',
'ComentarioVisita',
'GestorOrigen',
'IdGeografico',
'NombApellido',
'Telefonouno',
'Telefonodos',
'Correo',
'Direccion',
'Distrito',
'Provincia',
'CCOriginal',
'EstadoTelecierre',
'TCGestor',
'TCMotivoNoVenta',
'TCTipoVenta',
'Checkin',
'LeadID',
'FechaHoraInst',
'TipoDocumento',
'Documento',
'Accion',
'PerfilScoring',
'IdConsultaScoring',
'DistritalScoring',
'ProvincialScoring',
'DepartamentalScoring',
'MatriculaScoring',
'Score',
'PredictorIngresos_IP',
'CantidadDocumentosMorosos',
'FechaHoraConsulta',
'Zona',
'Seccion',
'IMP'];
        $this->reportWriter->writeHeaderRow($headerRow);
    }
    
    //Funciones de Descarga VisitasRE_Scoring
    public function fetchDataAndCreateReport_VisitasRE_Scoring($outputPath,$idVenta){
           $this->reportWriter = $this->createReportWriter($outputPath);
        $this->writeReportHeader_VisitasRE_Scoring();

        // Make sure to only select the fields we are interested in
        $query = "SELECT 
                v.IdVenta,
                v.IdUsuario,
                v.IdCliente,
                v.Matricula,
                v.Comercial,
                v.CC,
                v.JefeVenta,
                v.CategoriaVenta,
                v.TipoVenta,
                v.Segmento,
                v.NProspecto,
                v.NContrato,
                v.PrecioKit,
                v.TotalExtra,
                v.IngresoKIT,
                v.IngresoExtras,
                v.IngresoTotal,
                v.EstadoVenta,
                v.ExtraAutorizado,
                v.MesesAutorizado,
                v.N590,
                v.FormaPago,
                v.PAI,
                v.ResumenVenta,
                v.FechaHoraREG,
                v.EstadoVisita,
                v.FechaVisita,
                v.HoraVisita,
                v.MesVisita,
                v.FVencimientoVisita,
                v.GestorMKT,
                v.OrigenVisita,
                v.ComentarioVisita,
                v.GestorOrigen,
                c.IdGeografico,
                c.NombApellido,
                c.Telefonouno,
                c.Telefonodos,
                c.Correo,
                c.Direccion,
                g.Distrito,
                g.Provincia,
                v.CCOriginal,
                v.EstadoTelecierre,
                v.TCGestor,
                v.TCMotivoNoVenta,
                v.TCTipoVenta,
                v.Checkin,
                v.LeadID,
                concat(a.FechaInst, ' ', a.HoraInst) FechaHoraInst,
                d.TipoDocumento,
                d.Documento,
                v.Accion,
                v.PerfilScoring,
                v.IdConsultaScoring,
                s.Distrital as DistritalScoring,
                s.Provincial as ProvincialScoring,
                s.Departamental as DepartamentalScoring,
                s.GestorOrigen as MatriculaScoring,
                s.Score,
                s.PredictorIngresos_IP,
                s.CantidadDocumentosMorosos,
                s.FechaHoraReg as FechaHoraConsulta,
                v.IMP,
                v.FObras,
                v.FVisitaObras,
                v.FechaHoraAsignacion,
                c.TipoSegmento,
                c.NPiso,
                c.TipoVial,
                gl.SubTipificacion,
                c.objGeofinder,
                v.MotivoAsignacion,
                v.ContactoAsignacion,
                v.CumpleRanking,
                v.MatriculaOriginal,
                v.visePiso,
                v.rubroNeg,
                v.puertaCalle,
                v.nobleParedes,
                v.nobleTecho,
                v.metros,
                v.motivoInteres,
                v.conSistema,
                v.empresaSistema,
                v.bancarizado                  
                FROM yifi58ge_dbVENTAS.tblventa v INNER JOIN yifi58ge_dbVENTAS.tblcliente c on c.IdCliente=v.IdCliente
                INNER JOIN yifi58ge_dbVENTAS.tblagenda a ON a.IdVenta=v.IdVenta 
                INNER JOIN yifi58ge_dbVENTAS.tblgeografico g ON g.IdGeografico=c.IdGeografico
                LEFT JOIN yifi58ge_dbVENTAS.tbldocumento d ON c.IdDocVisita = d.IdDoc
                LEFT JOIN yifi58ge_dbVENTAS.tbltmpscoring s ON s.IdConsultaScoring=v.IdConsultaScoring
                LEFT JOIN yifi58ge_dbVENTAS.tblLeadCC l ON l.IdLeadCC=v.IdLead
                LEFT JOIN yifi58ge_dbVENTAS.tblGestionLead gl  ON l.UltimaGestion=gl.IdGestionLead
                WHERE (v.TipoVenta='RE' or v.TipoVenta='REGGCC')
                and v.IdVenta >=?";
                $array = [$idVenta];

        switch ($this->fetchingMethod) {
            case self::FETCH_ROWS_ONE_BY_ONE:
                $this->fetchRowsOneByOneAndWriteThem_VisitasRE_Scoring($query,$array);
                break;
            case self::FETCH_ROWS_IN_BATCH:
                $this->fetchRowsInBatchAndWriteThem($query);
                break;
            case self::FETCH_ROWS_ALL_AT_ONCE:
                $this->fetchAllRowsAtOnceAndWriteThem($query);
                break;
        }

        $this->reportWriter->close();
    }
    private function writeReportHeader_VisitasRE_Scoring(){
        // The header will be bold
        $headerRow = ['IdVenta',
                'IdUsuario',
                'IdCliente',
                'Matricula',
                'Comercial',
                'CC',
                'JefeVenta',
                'CategoriaVenta',
                'TipoVenta',
                'Segmento',
                'NProspecto',
                'NContrato',
                'PrecioKit',
                'TotalExtra',
                'IngresoKIT',
                'IngresoExtras',
                'IngresoTotal',
                'EstadoVenta',
                'ExtraAutorizado',
                'MesesAutorizado',
                'N590',
                'FormaPago',
                'PAI',
                'ResumenVenta',
                'FechaHoraREG',
                'EstadoVisita',
                'FechaVisita',
                'HoraVisita',
                'MesVisita',
                'FVencimientoVisita',
                'GestorMKT',
                'OrigenVisita',
                'ComentarioVisita',
                'GestorOrigen',
                'IdGeografico',
                'NombApellido',
                'Telefonouno',
                'Telefonodos',
                'Correo',
                'Direccion',
                'Distrito',
                'Provincia',
                'CCOriginal',
                'EstadoTelecierre',
                'TCGestor',
                'TCMotivoNoVenta',
                'TCTipoVenta',
                'Checkin',
                'LeadID',
                'FechaHoraInst',
                'TipoDocumento',
                'Documento',
                'Accion',
                'PerfilScoring',
                'IdConsultaScoring',
                'DistritalScoring',
                'ProvincialScoring',
                'DepartamentalScoring',
                'MatriculaScoring',
                'Score',
                'PredictorIngresos_IP',
                'CantidadDocumentosMorosos',
                'FechaHoraConsulta',
                'Zona',
                'Seccion',
                'IMP',
                'FObras',
                'FVisitaObras',
                'FechaHoraAsignacion',
                'TipoSegmento',
                'NPiso',
                'TipoVial',
                'SubTipificacion',
                'MotivoAsignacion',
                'ContactoAsignacion',
                'CumpleRanking',
                'MatriculaOriginal',
                 'visePiso',
                 'rubroNeg',
                 'puertaCalle',
                 'nobleParedes',
                 'nobleTecho',
                 'metros',
                 'motivoInteres',
                 'conSistema',
                 'empresaSistema',
                 'bancarizado'                 
                ];
        $this->reportWriter->writeHeaderRow($headerRow);
    }

   private function fetchRowsOneByOneAndWriteThem_VisitasRE_Scoring($query,$array=null){
        $dbRowIterator = new SingleFetchIterator($this->pdo, $query,$array);

        foreach ($dbRowIterator as $dbRow) {
                if(!empty($dbRow['objGeofinder']) || !is_null($dbRow['objGeofinder'])){
                    $objGeofinder = json_decode($dbRow['objGeofinder'], true);
                    //$InfoCheckIn = json_decode($Venta['InfoCheckIn'], true);
                    $zonas = $objGeofinder['zonas'];
                    //ZONA
                    if(!empty($zonas[0]['valor'])){
                       $zona = $zonas[0]['valor']; 
                    }else{
                      $zona = '';  
                    }
                    //MICROZONA
                    if(!empty($zonas[1]['valor'])){
                       $microzona = $zonas[1]['valor']; 
                    }else{
                      $microzona = '';  
                    }                    
                }else{
                    //ZONA
                    $zona = '';
                    //MICROZONA
                    $microzona='';
                }
        $reportRow = [$dbRow['IdVenta'],
                    $dbRow['IdUsuario'],
                    $dbRow['IdCliente'],
                    $dbRow['Matricula'],
                    $dbRow['Comercial'],
                    $dbRow['CC'],
                    $dbRow['JefeVenta'],
                    $dbRow['CategoriaVenta'],
                    $dbRow['TipoVenta'],
                    $dbRow['Segmento'],
                    $dbRow['NProspecto'],
                    $dbRow['NContrato'],
                    $dbRow['PrecioKit'],
                    $dbRow['TotalExtra'],
                    $dbRow['IngresoKIT'],
                    $dbRow['IngresoExtras'],
                    $dbRow['IngresoTotal'],
                    $dbRow['EstadoVenta'],
                    $dbRow['ExtraAutorizado'],
                    $dbRow['MesesAutorizado'],
                    $dbRow['N590'],
                    $dbRow['FormaPago'],
                    $dbRow['PAI'],
                    $dbRow['ResumenVenta'],
                    $dbRow['FechaHoraREG'],
                    $dbRow['EstadoVisita'],
                    $dbRow['FechaVisita'],
                    $dbRow['HoraVisita'],
                    $dbRow['MesVisita'],
                    $dbRow['FVencimientoVisita'],
                    $dbRow['GestorMKT'],
                    $dbRow['OrigenVisita'],
                    $dbRow['ComentarioVisita'],
                    $dbRow['GestorOrigen'],
                    $dbRow['IdGeografico'],
                    $dbRow['NombApellido'],
                    $dbRow['Telefonouno'],
                    $dbRow['Telefonodos'],
                    $dbRow['Correo'],
                    $dbRow['Direccion'],
                    $dbRow['Distrito'],
                    $dbRow['Provincia'],
                    $dbRow['CCOriginal'],
                    $dbRow['EstadoTelecierre'],
                    $dbRow['TCGestor'],
                    $dbRow['TCMotivoNoVenta'],
                    $dbRow['TCTipoVenta'],
                    $dbRow['Checkin'],
                    $dbRow['LeadID'],
                    $dbRow['FechaHoraInst'],
                    $dbRow['TipoDocumento'],
                    $dbRow['Documento'],
                    $dbRow['Accion'],
                    $dbRow['PerfilScoring'],
                    $dbRow['IdConsultaScoring'],
                    $dbRow['DistritalScoring'],
                    $dbRow['ProvincialScoring'],
                    $dbRow['DepartamentalScoring'],
                    $dbRow['MatriculaScoring'],
                    $dbRow['Score'],
                    $dbRow['PredictorIngresos_IP'],
                    $dbRow['CantidadDocumentosMorosos'],
                    $dbRow['FechaHoraConsulta'],
                    $zona,
                    $microzona,
                    $dbRow['IMP'],
                    $dbRow['FObras'],
                    $dbRow['FVisitaObras'],
                    $dbRow['FechaHoraAsignacion'],
                    $dbRow['TipoSegmento'],
                    $dbRow['NPiso'],
                    $dbRow['TipoVial'],
                    $dbRow['SubTipificacion'],
                    $dbRow['MotivoAsignacion'],
                    $dbRow['ContactoAsignacion'],
                    $dbRow['CumpleRanking'],
                    $dbRow['MatriculaOriginal'],
                     $dbRow['visePiso'],
                     $dbRow['rubroNeg'],
                     $dbRow['puertaCalle'],
                     $dbRow['nobleParedes'],
                     $dbRow['nobleTecho'],
                     $dbRow['metros'],
                     $dbRow['motivoInteres'],
                     $dbRow['conSistema'],
                     $dbRow['empresaSistema'],
                     $dbRow['bancarizado'],                     
                    ];
            $this->reportWriter->writeRow($reportRow);
        }
    }    
    // 
    
    //Funciones de Descarga de Retencion
     public function fetchDataAndCreateReport_Retencion($outputPath){
           $this->reportWriter = $this->createReportWriter($outputPath);
        $this->writeReportHeader_Retencion();

        // Make sure to only select the fields we are interested in
        $query = "
        SELECT
        r.IdRetencion,
        c.NCliente,
        c.NombApellido Cliente,
        c.Telefonouno,
        c.Correo,
        c.Direccion,
        r.IdCliente,
        r.MotivoAviso,
        r.Gestor,
        r.FechaCierre,
        r.TipoCierre,
        r.MotivoCierre,
        r.EstadoFinGestion,
        r.FechaHoraREG,
        r.TipoCanal,
        r.DuracionOferta,
        r.Reincidente,
        r.MotivoCierreTC,
        r.Competencia
        FROM yifi58ge_dbVENTAS.tblretencion r
        INNER JOIN yifi58ge_dbVENTAS.tblcliente c on c.IdCliente=r.IdCliente
        INNER JOIN yifi58ge_dbVENTAS.tblgeografico g on g.IdGeografico=c.IdGeografico
        ORDER BY  r.IdRetencion DESC
        ";

        switch ($this->fetchingMethod) {
            case self::FETCH_ROWS_ONE_BY_ONE:
                $this->fetchRowsOneByOneAndWriteThem_Retencion($query);
                break;
            case self::FETCH_ROWS_IN_BATCH:
                $this->fetchRowsInBatchAndWriteThem($query);
                break;
            case self::FETCH_ROWS_ALL_AT_ONCE:
                $this->fetchAllRowsAtOnceAndWriteThem($query);
                break;
        }

        $this->reportWriter->close();
    }
    private function writeReportHeader_Retencion(){
        // The header will be bold
        $headerRow = ['IdRetencion',
        'NCliente',
        'Cliente',
        'Telefonouno',
        'Correo',
        'Direccion',
        'IdCliente',
        'MotivoAviso',
        'Gestor',
        'FechaCierre',
        'TipoCierre',
        'MotivoCierre',
        'EstadoFinGestion',
        'FechaHoraREG',
        'TipoCanal',
        'DuracionOferta',
        'Reincidente',
        'MotivoCierreTC',
        'Competencia'];
        $this->reportWriter->writeHeaderRow($headerRow);
    }
     private function fetchRowsOneByOneAndWriteThem_Retencion($query){
        $dbRowIterator = new SingleFetchIterator($this->pdo, $query);

        foreach ($dbRowIterator as $dbRow) {

            $reportRow = [
                    $dbRow['IdRetencion'],
                    $dbRow['NCliente'],
                    $dbRow['Cliente'],
                    $dbRow['Telefonouno'],
                    $dbRow['Correo'],
                    $dbRow['Direccion'],
                    $dbRow['IdCliente'],
                    $dbRow['MotivoAviso'],
                    $dbRow['Gestor'],
                    $dbRow['FechaCierre'],
                    $dbRow['TipoCierre'],
                    $dbRow['MotivoCierre'],
                    $dbRow['EstadoFinGestion'],
                    $dbRow['FechaHoraREG'],
                    $dbRow['TipoCanal'],
                    $dbRow['DuracionOferta'],
                    $dbRow['Reincidente'],
                    $dbRow['MotivoCierreTC'],
                    $dbRow['Competencia']
                    ];
            $this->reportWriter->writeRow($reportRow);
        }
    } 
    
    
        //Funciones de Descarga de 590Online
     public function fetchDataAndCreateReport_590Online($outputPath,$mesactual){
        $this->reportWriter = $this->createReportWriter($outputPath);
        $this->writeReportHeader_590Online();

        // Make sure to only select the fields we are interested in
        $query = "
        SELECT 
v.IdVenta,
v.NContrato as Instalacion,
v.NProspecto as Prospecto,
v.TipoKIT,
v.Matricula as Vendedor,
a.MatriculaTec as Instalador,
v.TipoVenta,
v.Segmento,
v.MesesAutorizado as Oferta,
v.IngresoKIT as KIT,
v.IngresoExtras as Extras,
v.IngresoTotal as TOTAL,
v.CC,
f.TipoComprobante,
f.Cod_WO_Inst as PayCode_WO,
v.FormaPagoCuota as PayCode_CTR,
f.InfoGroup,
v.IdGGCC as Member,
a.FechaInst as Fecha_Instalacion,
f.FechaFacturacion as Fecha_Facturacion,
v.PAI as PAI36X30,
v.ScoringRP as Scoring,
v.PrecioEspecial as Renove_o_Ice_Breaker,
f.Comentario as ComentarioFact,
f.NCarrera,
sc.NumeroDocumento as tax_no,
CASE WHEN sc.TipoDocumento=6 THEN sc.RazonSocial ELSE CONCAT(sc.Nombres,' ',sc.ApellidoPaterno,' ',sc.ApellidoMaterno) END as Nombre_RazonSocial,
v.ExtraAutorizado,
v.EstadoVenta,
v.N590,
v.ResumenVenta,
v.PrecioKit,
v.TotalExtra,
v.TCGestor
FROM yifi58ge_dbVENTAS.tblventa v 
INNER JOIN yifi58ge_dbVENTAS.tblcliente c on c.IdCliente=v.IdCliente
INNER JOIN yifi58ge_dbVENTAS.tblgeografico g on g.IdGeografico=c.IdGeografico
INNER JOIN yifi58ge_dbVENTAS.tblagenda a on a.IdVenta=v.IdVenta
INNER JOIN yifi58ge_dbVENTAS.tblfacturacion f on f.IdVenta=v.IdVenta
LEFT JOIN yifi58ge_dbVENTAS.tbltmpscoring sc on sc.IdConsultaScoring=v.IdConsultaScoringRP
WHERE (v.EstadoVenta='PorFacturar' or v.EstadoVenta='Facturada') and v.N590='$mesactual'";

        switch ($this->fetchingMethod) {
            case self::FETCH_ROWS_ONE_BY_ONE:
                $this->fetchRowsOneByOneAndWriteThem_590Online($query);
                break;
            case self::FETCH_ROWS_IN_BATCH:
                $this->fetchRowsInBatchAndWriteThem($query);
                break;
            case self::FETCH_ROWS_ALL_AT_ONCE:
                $this->fetchAllRowsAtOnceAndWriteThem($query);
                break;
        }

        $this->reportWriter->close();
    }
    private function writeReportHeader_590Online(){
        // The header will be bold
        $headerRow = ['IdVenta',
            'Instalacion',
            'Prospecto',
            'TipoKIT',
            'Vendedor',
            'Instalador',
            'TipoVenta',
            'Segmento',
            'Oferta',
            'KIT',
            'Extras',
            'TOTAL',
            'CC',
            'TipoComprobante',
            'PayCode_WO',
            'PayCode_CTR',
            'InfoGroup',
            'Member',
            'Fecha_Instalacion',
            'Fecha_Facturacion',
            'PAI36X30',
            'Scoring',
            'Renove_o_Ice_Breaker',
            'ComentarioFact',
            'NCarrera',
            'tax_no',
            'Nombre_RazonSocial',
            'ExtraAutorizado',
            'EstadoVenta',
            'N590',
            'ResumenVenta',
            'PrecioKit',
            'TotalExtra',
            'TCGestor'];
                    $this->reportWriter->writeHeaderRow($headerRow);
                }
                 private function fetchRowsOneByOneAndWriteThem_590Online($query){
                    $dbRowIterator = new SingleFetchIterator($this->pdo, $query);
            
                    foreach ($dbRowIterator as $dbRow) {
            
                        $reportRow = [
                                $dbRow['IdVenta'],
            $dbRow['Instalacion'],
            $dbRow['Prospecto'],
            $dbRow['TipoKIT'],
            $dbRow['Vendedor'],
            $dbRow['Instalador'],
            $dbRow['TipoVenta'],
            $dbRow['Segmento'],
            $dbRow['Oferta'],
            $dbRow['KIT'],
            $dbRow['Extras'],
            $dbRow['TOTAL'],
            $dbRow['CC'],
            $dbRow['TipoComprobante'],
            $dbRow['PayCode_WO'],
            $dbRow['PayCode_CTR'],
            $dbRow['InfoGroup'],
            $dbRow['Member'],
            $dbRow['Fecha_Instalacion'],
            $dbRow['Fecha_Facturacion'],
            $dbRow['PAI36X30'],
            $dbRow['Scoring'],
            $dbRow['Renove_o_Ice_Breaker'],
            $dbRow['ComentarioFact'],
            $dbRow['NCarrera'],
            $dbRow['tax_no'],
            $dbRow['Nombre_RazonSocial'],
            $dbRow['ExtraAutorizado'],
            $dbRow['EstadoVenta'],
            $dbRow['N590'],
            $dbRow['ResumenVenta'],
            $dbRow['PrecioKit'],
            $dbRow['TotalExtra'],
            $dbRow['TCGestor']
                    ];
            $this->reportWriter->writeRow($reportRow);
        }
    } 
    

    //Funciones de Descarga VisitasRE_Reasignaciones
    public function fetchDataAndCreateReport_Visitas_Reasignaciones($outputPath){
           $this->reportWriter = $this->createReportWriter($outputPath);
        $this->writeReportHeader_Visitas_Reasignaciones();

        // Make sure to only select the fields we are interested in
        $query = "SELECT 
        r.IdReasignacion,
        r.IdVenta,
        r.Matricula,
        r.Categoria,
        r.FechaAsignacion,
        r.CC,
        r.EstadoAsignacion,
        v.FechaVisita,
        v.HoraVisita,
        v.EstadoVenta,
        v.Checkin,
        v.InfoCheckIn,
        v.NProspecto,
        v.FechaHoraREG,
        r.AsignacionPlantilla,
        c.Correo,
        r.ObjCorreoM
        FROM yifi58ge_dbVENTAS.tblreasignacion r INNER JOIN tblventa v on v.IdVenta=r.IdVenta
        INNER JOIN tblcliente c on c.IdCliente=v.IdCliente";

        switch ($this->fetchingMethod) {
            case self::FETCH_ROWS_ONE_BY_ONE:
                $this->fetchRowsOneByOneAndWriteThem_Visitas_Reasignaciones($query);
                break;
            case self::FETCH_ROWS_IN_BATCH:
                $this->fetchRowsInBatchAndWriteThem($query);
                break;
            case self::FETCH_ROWS_ALL_AT_ONCE:
                $this->fetchAllRowsAtOnceAndWriteThem($query);
                break;
        }

        $this->reportWriter->close();
    }
    private function writeReportHeader_Visitas_Reasignaciones(){
        // The header will be bold
        $headerRow = ['IdReasignacion',
            'IdVenta',
            'Matricula',
            'Categoria',
            'FechaAsignacion',
            'CC',
            'EstadoAsignacion',
            'FechaVisita',
            'HoraVisita',
            'EstadoVenta',
            'Checkin',
            'FechaEnvioCheckin',
            'MatriculaCheckin',
            'TipoCheckin',
            'NProspecto',
            'FechaHoraREG',
            'AsignacionPlantilla',
            'Correo',
            'IdMail'
        ];
        $this->reportWriter->writeHeaderRow($headerRow);
    }

   private function fetchRowsOneByOneAndWriteThem_Visitas_Reasignaciones($query){
        $dbRowIterator = new SingleFetchIterator($this->pdo, $query);

        foreach ($dbRowIterator as $dbRow) {
                if(!empty($dbRow['InfoCheckIn']) || !is_null($dbRow['InfoCheckIn'])){
                    $InfoCheckIn = json_decode($dbRow['InfoCheckIn'], true);
                    $FechaEnvioCheckin = $InfoCheckIn['FechaEnvio'];
                    if(!empty($FechaEnvioCheckin)){
                       $FechaEnvioCheckin = $FechaEnvioCheckin; 
                    }else{
                      $FechaEnvioCheckin = '';  
                    }
                    
                    $MatriculaCheckin = $InfoCheckIn['Matricula'];
                    if(!empty($MatriculaCheckin)){
                       $MatriculaCheckin = $MatriculaCheckin; 
                    }else{
                      $MatriculaCheckin = '';  
                    }
                    $TipoCheckin = $InfoCheckIn['Tipo'];
                    if(!empty($TipoCheckin)){
                       $TipoCheckin = $TipoCheckin; 
                    }else{
                      $TipoCheckin = '';  
                    }                       
                    
                }else{
                    $FechaEnvioCheckin = '';
                    $MatriculaCheckin = '';
                    $TipoCheckin = '';
                }
                
                if(!empty($dbRow['ObjCorreoM']) || !is_null($dbRow['ObjCorreoM'])){
                    $ObjCorreoM = json_decode($dbRow['ObjCorreoM'], true);
                    $IdMail = $ObjCorreoM['Data']['Mails'][0]['Id'];
                    if(!empty($IdMail)){
                       $IdMail = $IdMail; 
                    }else{
                      $IdMail = '';  
                    }                    
                }
                else{
                    $IdMail = '';  
                }
                
                
        $reportRow = [$dbRow['IdReasignacion'],
                    $dbRow['IdVenta'],
                    $dbRow['Matricula'],
                    $dbRow['Categoria'],
                    $dbRow['FechaAsignacion'],
                    $dbRow['CC'],
                    $dbRow['EstadoAsignacion'],
                    $dbRow['FechaVisita'],
                    $dbRow['HoraVisita'],
                    $dbRow['EstadoVenta'],
                    $dbRow['Checkin'],
                    $FechaEnvioCheckin,
                    $MatriculaCheckin,
                    $TipoCheckin,
                    $dbRow['NProspecto'],
                    $dbRow['FechaHoraREG'],
                    $dbRow['AsignacionPlantilla'],
                    $dbRow['Correo'],
                    $IdMail
                    ];
            $this->reportWriter->writeRow($reportRow);
        }
    }    
    // 
    
    //Funciones de Descargar VisitasRE_Reasignaciones_150
    //fetchDataAndCreateReport_Visitas_Reasignaciones150
    public function fetchDataAndCreateReport_Visitas_Reasignaciones150($outputPath){
           $this->reportWriter = $this->createReportWriter($outputPath);
        $this->writeReportHeader_Visitas_Reasignaciones150();

        // Make sure to only select the fields we are interested in
        $query = "SELECT 
        r.IdReasignacion,
        r.IdVenta,
        r.Matricula,
        r.Categoria,
        r.FechaAsignacion,
        r.CC,
        r.EstadoAsignacion,
        v.FechaVisita,
        v.HoraVisita,
        v.EstadoVenta,
        v.Checkin,
        v.InfoCheckIn,
        v.NProspecto,
        v.FechaHoraREG,
        r.AsignacionPlantilla,
        c.Correo,
        r.ObjCorreoM
        FROM yifi58ge_dbVENTAS.tblreasignacion r INNER JOIN tblventa v on v.IdVenta=r.IdVenta
        INNER JOIN tblcliente c on c.IdCliente=v.IdCliente
        where r.CC='150'
        ORDER BY v.NProspecto DESC,r.IdReasignacion ASC        
        ";

        switch ($this->fetchingMethod) {
            case self::FETCH_ROWS_ONE_BY_ONE:
                $this->fetchRowsOneByOneAndWriteThem_Visitas_Reasignaciones150($query);
                break;
            case self::FETCH_ROWS_IN_BATCH:
                $this->fetchRowsInBatchAndWriteThem($query);
                break;
            case self::FETCH_ROWS_ALL_AT_ONCE:
                $this->fetchAllRowsAtOnceAndWriteThem($query);
                break;
        }

        $this->reportWriter->close();
    }
    private function writeReportHeader_Visitas_Reasignaciones150(){
        // The header will be bold
        $headerRow = ['IdReasignacion',
            'IdVenta',
            'Matricula',
            'Categoria',
            'FechaAsignacion',
            'CC',
            'EstadoAsignacion',
            'FechaVisita',
            'HoraVisita',
            'EstadoVenta',
            'Checkin',
            'FechaEnvioCheckin',
            'MatriculaCheckin',
            'TipoCheckin',
            'NProspecto',
            'FechaHoraREG',
            'AsignacionPlantilla',
            'Correo',
            'IdMail'
        ];
        $this->reportWriter->writeHeaderRow($headerRow);
    }

   private function fetchRowsOneByOneAndWriteThem_Visitas_Reasignaciones150($query){
        $dbRowIterator = new SingleFetchIterator($this->pdo, $query);

        foreach ($dbRowIterator as $dbRow) {
                if(!empty($dbRow['InfoCheckIn']) || !is_null($dbRow['InfoCheckIn'])){
                    $InfoCheckIn = json_decode($dbRow['InfoCheckIn'], true);
                    $FechaEnvioCheckin = $InfoCheckIn['FechaEnvio'];
                    if(!empty($FechaEnvioCheckin)){
                       $FechaEnvioCheckin = $FechaEnvioCheckin; 
                    }else{
                      $FechaEnvioCheckin = '';  
                    }
                    
                    $MatriculaCheckin = $InfoCheckIn['Matricula'];
                    if(!empty($MatriculaCheckin)){
                       $MatriculaCheckin = $MatriculaCheckin; 
                    }else{
                      $MatriculaCheckin = '';  
                    }
                    $TipoCheckin = $InfoCheckIn['Tipo'];
                    if(!empty($TipoCheckin)){
                       $TipoCheckin = $TipoCheckin; 
                    }else{
                      $TipoCheckin = '';  
                    }                       
                    
                }else{
                    $FechaEnvioCheckin = '';
                    $MatriculaCheckin = '';
                    $TipoCheckin = '';
                }
                
                if(!empty($dbRow['ObjCorreoM']) || !is_null($dbRow['ObjCorreoM'])){
                    $ObjCorreoM = json_decode($dbRow['ObjCorreoM'], true);
                    $IdMail = $ObjCorreoM['Data']['Mails'][0]['Id'];
                    if(!empty($IdMail)){
                       $IdMail = $IdMail; 
                    }else{
                      $IdMail = '';  
                    }                    
                }
                else{
                    $IdMail = '';  
                }
                
                
        $reportRow = [$dbRow['IdReasignacion'],
                    $dbRow['IdVenta'],
                    $dbRow['Matricula'],
                    $dbRow['Categoria'],
                    $dbRow['FechaAsignacion'],
                    $dbRow['CC'],
                    $dbRow['EstadoAsignacion'],
                    $dbRow['FechaVisita'],
                    $dbRow['HoraVisita'],
                    $dbRow['EstadoVenta'],
                    $dbRow['Checkin'],
                    $FechaEnvioCheckin,
                    $MatriculaCheckin,
                    $TipoCheckin,
                    $dbRow['NProspecto'],
                    $dbRow['FechaHoraREG'],
                    $dbRow['AsignacionPlantilla'],
                    $dbRow['Correo'],
                    $IdMail
                    ];
            $this->reportWriter->writeRow($reportRow);
        }
    }        
    // fin Reasignaciones150
    
     //Funciones de Descarga Ventas_Scoring_PROy
         public function fetchDataAndCreateReport_Ventas_Scoring_Proy($outputPath,$mesactual){
           $this->reportWriter = $this->createReportWriter($outputPath);
        $this->writeReportHeader_Ventas_Scoring_Proy();

        // Make sure to only select the fields we are interested in
        $query = "SELECT
v.IdVenta,
v.IdUsuario,
v.IdCliente,
v.Matricula,
v.Comercial,
v.CC,
v.JefeVenta,
v.CategoriaVenta,
v.TipoVenta,
v.Segmento,
v.NProspecto,
v.NContrato,
v.PrecioKit,
v.TotalExtra,
v.IngresoKIT,
v.IngresoExtras,
v.IngresoTotal,
v.EstadoVenta,
v.ExtraAutorizado,
v.MesesAutorizado,
v.N590,
v.FormaPago,
v.PAI,
v.ResumenVenta,
v.FechaHoraREG,
v.EstadoVisita,
v.FechaVisita,
v.HoraVisita,
v.GestorMKT,
v.OrigenVisita,
v.GestorOrigen,
c.IdGeografico,
g.Distrito,
g.Provincia,
v.CCOriginal,
v.IMP,
v.MesVisita,
a.FechaInst,
a.HoraInst,
a.InstaladoX,
v.IdProspectoRP,
f.FechaFacturacion,
v.ScoringRP as ScoringRP_Result,
v.TipoKIT,
v.CantidadExtras,
scven.Accion as AccionRP,
scven.PerfilScoring as PerfilScoringRP,
v.IdConsultaScoringRP,
scven.Score as ScoreRP,
scven.PredictorIngresos_IP as PredictorIngresos_IPRP,
scven.CantidadDocumentosMorosos as CantidadDocumentosMorososRP,
scven.FechaHoraReg as FechaHoraConsultaRP,
scven.TipoDocumento as TipoDocumentoRP,
scven.NumeroDocumento as NumeroDocumentoRP,
scven.Segmento as SegmentoRP,
scven.NodoDescripcion as NodoDescripcionRP,
scven.NodoAccion as NodoAccionRP,
v.Accion as AccionMKT,
v.PerfilScoring as PerfilScoringMKT,
v.IdConsultaScoring as IdConsultaScoringMKT,
scmkt.Score as ScoreMKT,
scmkt.PredictorIngresos_IP as PredictorIngresos_IPMKT,
scmkt.CantidadDocumentosMorosos as CantidadDocumentosMorososMKT,
scmkt.FechaHoraReg as FechaHoraConsultaMKT
from tblventa v
INNER JOIN tblcliente c on c.IdCliente=v.IdCliente
INNER JOIN tblgeografico g on g.IdGeografico=c.IdGeografico
INNER JOIN tblagenda a on a.IdVenta=v.IdVenta
INNER JOIN tblfacturacion f on f.IdVenta=v.IdVenta
LEFT JOIN tbltmpscoring scven on scven.IdConsultaScoring=v.IdConsultaScoringRP
LEFT JOIN tbltmpscoring scmkt on scmkt.IdConsultaScoring=v.IdConsultaScoring 
WHERE v.N590='".$mesactual."'";

        switch ($this->fetchingMethod) {
            case self::FETCH_ROWS_ONE_BY_ONE:
                $this->fetchRowsOneByOneAndWriteThem_Ventas_Scoring_Proy($query);
                break;
            case self::FETCH_ROWS_IN_BATCH:
                $this->fetchRowsInBatchAndWriteThem($query);
                break;
            case self::FETCH_ROWS_ALL_AT_ONCE:
                $this->fetchAllRowsAtOnceAndWriteThem($query);
                break;
        }

        $this->reportWriter->close();
    }
    private function writeReportHeader_Ventas_Scoring_Proy(){
        // The header will be bold
        $headerRow = ['IdVenta',
'IdUsuario',
'IdCliente',
'Matricula',
'Comercial',
'CC',
'JefeVenta',
'CategoriaVenta',
'TipoVenta',
'Segmento',
'NProspecto',
'NContrato',
'PrecioKit',
'TotalExtra',
'IngresoKIT',
'IngresoExtras',
'IngresoTotal',
'EstadoVenta',
'ExtraAutorizado',
'MesesAutorizado',
'N590',
'FormaPago',
'PAI',
'ResumenVenta',
'FechaHoraREG',
'EstadoVisita',
'FechaVisita',
'HoraVisita',
'GestorMKT',
'OrigenVisita',
'GestorOrigen',
'IdGeografico',
'Distrito',
'Provincia',
'CCOriginal',
'IMP',
'MesVisita',
'FechaInst',
'HoraInst',
'InstaladoX',
'IdProspectoRP',
'FechaFacturacion',
'ScoringRP_Result',
'TipoKIT',
'CantidadExtras',
'AccionRP',
'PerfilScoringRP',
'IdConsultaScoringRP',
'ScoreRP',
'PredictorIngresos_IPRP',
'CantidadDocumentosMorososRP',
'FechaHoraConsultaRP',
'TipoDocumentoRP',
'NumeroDocumentoRP',
'SegmentoRP',
'NodoDescripcionRP',
'NodoAccionRP',
'AccionMKT',
'PerfilScoringMKT',
'IdConsultaScoringMKT',
'ScoreMKT',
'PredictorIngresos_IPMKT',
'CantidadDocumentosMorososMKT',
'FechaHoraConsultaMKT'
        ];
        $this->reportWriter->writeHeaderRow($headerRow);
    }

   private function fetchRowsOneByOneAndWriteThem_Ventas_Scoring_Proy($query){
 $dbRowIterator = new SingleFetchIterator($this->pdo, $query);
            
                    foreach ($dbRowIterator as $dbRow) {
            
                        $reportRow = [
                            $dbRow['IdVenta'],
                            $dbRow['IdUsuario'],
                            $dbRow['IdCliente'],
                            $dbRow['Matricula'],
                            $dbRow['Comercial'],
                            $dbRow['CC'],
                            $dbRow['JefeVenta'],
                            $dbRow['CategoriaVenta'],
                            $dbRow['TipoVenta'],
                            $dbRow['Segmento'],
                            $dbRow['NProspecto'],
                            $dbRow['NContrato'],
                            $dbRow['PrecioKit'],
                            $dbRow['TotalExtra'],
                            $dbRow['IngresoKIT'],
                            $dbRow['IngresoExtras'],
                            $dbRow['IngresoTotal'],
                            $dbRow['EstadoVenta'],
                            $dbRow['ExtraAutorizado'],
                            $dbRow['MesesAutorizado'],
                            $dbRow['N590'],
                            $dbRow['FormaPago'],
                            $dbRow['PAI'],
                            $dbRow['ResumenVenta'],
                            $dbRow['FechaHoraREG'],
                            $dbRow['EstadoVisita'],
                            $dbRow['FechaVisita'],
                            $dbRow['HoraVisita'],
                            $dbRow['GestorMKT'],
                            $dbRow['OrigenVisita'],
                            $dbRow['GestorOrigen'],
                            $dbRow['IdGeografico'],
                            $dbRow['Distrito'],
                            $dbRow['Provincia'],
                            $dbRow['CCOriginal'],
                            $dbRow['IMP'],
                            $dbRow['MesVisita'],
                            $dbRow['FechaInst'],
                            $dbRow['HoraInst'],
                            $dbRow['InstaladoX'],
                            $dbRow['IdProspectoRP'],
                            $dbRow['FechaFacturacion'],
                            $dbRow['ScoringRP_Result'],
                            $dbRow['TipoKIT'],
                            $dbRow['CantidadExtras'],
                            $dbRow['AccionRP'],
                            $dbRow['PerfilScoringRP'],
                            $dbRow['IdConsultaScoringRP'],
                            $dbRow['ScoreRP'],
                            $dbRow['PredictorIngresos_IPRP'],
                            $dbRow['CantidadDocumentosMorososRP'],
                            $dbRow['FechaHoraConsultaRP'],
                            $dbRow['TipoDocumentoRP'],
                            $dbRow['NumeroDocumentoRP'],
                            $dbRow['SegmentoRP'],
                            $dbRow['NodoDescripcionRP'],
                            $dbRow['NodoAccionRP'],
                            $dbRow['AccionMKT'],
                            $dbRow['PerfilScoringMKT'],
                            $dbRow['IdConsultaScoringMKT'],
                            $dbRow['ScoreMKT'],
                            $dbRow['PredictorIngresos_IPMKT'],
                            $dbRow['CantidadDocumentosMorososMKT'],
                            $dbRow['FechaHoraConsultaMKT']
                    ];
            $this->reportWriter->writeRow($reportRow);
        }
    }    
    // 
   
       
     //Funciones de Descarga Ventas_GGCC
         public function fetchDataAndCreateReport_Ventas_GGCC($outputPath){
           $this->reportWriter = $this->createReportWriter($outputPath);
        $this->writeReportHeader_Ventas_GGCC();

        // Make sure to only select the fields we are interested in
        $query = "SELECT 
IdVenta,
EstadoVenta,
TipoVenta,
DC,
Comercial,
Segmento,
NProspecto,
NContrato,
Cliente,
CategoriaVenta,
PrecioKit,
TotalExtra,
ExtraAutorizado,
PAI,
N590,
FechaHoraREG,
FechaVisita,
Distrito,
InstaladoX,
FechaInst,
HoraInst,
ScoringRP,
IdGGCC
 FROM ggcccheck";

        switch ($this->fetchingMethod) {
            case self::FETCH_ROWS_ONE_BY_ONE:
                $this->fetchRowsOneByOneAndWriteThem_Ventas_GGCC($query);
                break;
            case self::FETCH_ROWS_IN_BATCH:
                $this->fetchRowsInBatchAndWriteThem($query);
                break;
            case self::FETCH_ROWS_ALL_AT_ONCE:
                $this->fetchAllRowsAtOnceAndWriteThem($query);
                break;
        }

        $this->reportWriter->close();
    }
    private function writeReportHeader_Ventas_GGCC(){
        // The header will be bold
        $headerRow = ['IdVenta',
            'EstadoVenta',
            'TipoVenta',
            'DC',
            'Comercial',
            'Segmento',
            'NProspecto',
            'NContrato',
            'Cliente',
            'CategoriaVenta',
            'PrecioKit',
            'TotalExtra',
            'ExtraAutorizado',
            'PAI',
            'N590',
            'FechaHoraREG',
            'FechaVisita',
            'Distrito',
            'InstaladoX',
            'FechaInst',
            'HoraInst',
            'ScoringRP',
            'IdGGCC'
        ];
        $this->reportWriter->writeHeaderRow($headerRow);
    }

   private function fetchRowsOneByOneAndWriteThem_Ventas_GGCC($query){
 $dbRowIterator = new SingleFetchIterator($this->pdo, $query);
            
                    foreach ($dbRowIterator as $dbRow) {
            
                        $reportRow = [
                        $dbRow['IdVenta'],
                        $dbRow['EstadoVenta'],
                        $dbRow['TipoVenta'],
                        $dbRow['DC'],
                        $dbRow['Comercial'],
                        $dbRow['Segmento'],
                        $dbRow['NProspecto'],
                        $dbRow['NContrato'],
                        $dbRow['Cliente'],
                        $dbRow['CategoriaVenta'],
                        $dbRow['PrecioKit'],
                        $dbRow['TotalExtra'],
                        $dbRow['ExtraAutorizado'],
                        $dbRow['PAI'],
                        $dbRow['N590'],
                        $dbRow['FechaHoraREG'],
                        $dbRow['FechaVisita'],
                        $dbRow['Distrito'],
                        $dbRow['InstaladoX'],
                        $dbRow['FechaInst'],
                        $dbRow['HoraInst'],
                        $dbRow['ScoringRP'],
                        $dbRow['IdGGCC']
                    ];
            $this->reportWriter->writeRow($reportRow);
        }
    }    
    // 

     //Funciones de Descarga LEAD
         public function fetchDataAndCreateReport_LEAD($outputPath,$fechagrabarfinicial,$fechagrabarffinal){
           $this->reportWriter = $this->createReportWriter($outputPath);
        $this->writeReportHeader_LEAD();

        // Make sure to only select the fields we are interested in
        $query = "SELECT 
                l.IdLeadCC as IDLead, 
                u.Matricula,
                Concat(u.Nombres,' ',u.Apellidos) as NombreGestor,
                l.FechaHoraReg,
                l.FechaHoraCotizacion,
                l.Gestion,
                l.ResultadoGestion,
                l.Campaña,
                l.Nombres,
                l.Apellidos,
                l.Telefono,
                l.Telefono2,
                l.Origen,
                l.LeadID,
                l.Departamento,
                l.Provincia,
                l.Distrito,
                l.Observaciones,
                l.Segmento,
                l.NLocales,
                l.GGCC,
                l.NDocumento,
                l.Vueltas,
                l.FechaUltimaGestion,
                l.Remarketing,
                uRMKT.Matricula as MatriculaRMKT,
                Concat(uRMKT.Nombres,' ',uRMKT.Apellidos) as RMKTNombreGestor,
                l.FechaHoraRemarketing,
                gl.Gestionable,
                gl.SubTipificacion,
                uGest.Matricula as ultimoGestor,
                Concat(uGest.Nombres,' ',uGest.Apellidos) as ultimoNombreGestor,
                d.Zonas,
                CASE
                    WHEN l.Gestion = 'inbound' THEN o.Canal
                ELSE 
	                CASE WHEN l.Origen ='GOOGLE_MARCA' THEN 'SEM MARCA' WHEN l.Origen ='SEO' THEN 'SEM MARCA' ELSE o.Canal END
                END AS Canal,
                v.IdVenta,
                v.TipoVenta,
                l.GestorTV,
                v.EstadoVisita,
                v.FechaHoraREG as FechaCreacionVisita,
                v.FechaVisita,
                v.NProspecto,
                c.Correo,
                l.TipoTransferencia,
                l.Gestionabilidad,
                l.ObsTransferencia,
                l.Contactado,
                l.DevolverLlamada,
                l.FechaUltimaGestionRemarketing,
                l.ResultadoGestionRemarketing,
                glr.SubTipificacion as SubtificacionRemarketing,
                l.visePiso,
                l.rubroNeg,
                l.puertaCalle,
                l.nobleParedes,
                l.nobleTecho,
                l.metros,
                l.motivoInteres,
                l.conSistema,
                l.empresaSistema,
                l.bancarizado,
                l.IdLeadCCREF
            FROM yifi58ge_dbVENTAS.tblLeadCC l
            INNER JOIN yifi58ge_dbUSUARIO.tblusuario u ON l.IdUsuario = u.IdUsuario
            LEFT JOIN yifi58ge_dbVENTAS.tblGestionLead gl on gl.IdGestionLead=l.UltimaGestion
            LEFT JOIN yifi58ge_dbUSUARIO.tblusuario uGest on gl.IdUsuario = uGest.IdUsuario
            LEFT JOIN yifi58ge_dbVENTAS.tbldistrital d ON d.CodUbigeoReniec=l.CodigoUbigeo
            LEFT JOIN yifi58ge_dbVENTAS.tblorigen o on o.Origen=l.Origen
            LEFT JOIN yifi58ge_dbUSUARIO.tblusuario uRMKT on l.IdUsuarioRemarketing = uRMKT.IdUsuario
            LEFT JOIN (SELECT * FROM yifi58ge_dbVENTAS.tblventa WHERE IdLead IS NOT NULL) v ON l.IdLeadCC = v.IdLead
            LEFT JOIN yifi58ge_dbVENTAS.tblcliente c on c.IdCliente=v.IdCliente
            LEFT JOIN yifi58ge_dbVENTAS.tblGestionLead glr on glr.IdGestionLead=l.UltimaGestionRemarketing
            where l.FechaHoraCotizacion BETWEEN '".$fechagrabarfinicial." 00:00:00' AND '".$fechagrabarffinal." 23:59:59'";

        switch ($this->fetchingMethod) {
            case self::FETCH_ROWS_ONE_BY_ONE:
                $this->fetchRowsOneByOneAndWriteThem_LEAD($query);
                break;
            case self::FETCH_ROWS_IN_BATCH:
                $this->fetchRowsInBatchAndWriteThem($query);
                break;
            case self::FETCH_ROWS_ALL_AT_ONCE:
                $this->fetchAllRowsAtOnceAndWriteThem($query);
                break;
        }

        $this->reportWriter->close();
    }
    private function writeReportHeader_LEAD(){
        // The header will be bold
    $headerRow = [
    'IDLead',
    'Matricula',
    'NombreGestor',
    'FechaHoraReg',
    'FechaHoraCotizacion',
    'Gestion',
    'ResultadoGestion',
    'Campaña',
    'Nombres',
    'Apellidos',
    'Telefono',
    'Telefono2',
    'Origen',
    'LeadID',
    'Departamento',
    'Provincia',
    'Distrito',
    'Observaciones',
    'Segmento',
    'NLocales',
    'GGCC',
    'NDocumento',
    'Vueltas',
    'FechaUltimaGestion',
    'Remarketing',
    'MatriculaRMKT',
    'RMKTNombreGestor',
    'FechaHoraRemarketing',
    'Gestionable',
    'SubTipificacion',
    'ultimoGestor',
    'ultimoNombreGestor',
    'Zonas',
    'Canal',
    'IdVenta',
    'TipoVenta',
    'GestorTV',
    'EstadoVisita',
    'FechaCreacionVisita',
    'FechaVisita',
    'NProspecto',
    'Correo',
    'TipoTransferencia',
    'Gestionabilidad',
    'ObsTransferencia',
    'Contactado',
    'DevolverLlamada',
    'FechaUltimaGestionRemarketing',
    'ResultadoGestionRemarketing',
    'SubtificacionRemarketing',
     'visePiso',
     'rubroNeg',
     'puertaCalle',
     'nobleParedes',
     'nobleTecho',
     'metros',
     'motivoInteres',
     'conSistema',
     'empresaSistema',
     'bancarizado',
     'IdLeadCCREF'
        ];
        $this->reportWriter->writeHeaderRow($headerRow);
    }

   private function fetchRowsOneByOneAndWriteThem_LEAD($query){
 $dbRowIterator = new SingleFetchIterator($this->pdo, $query);
            
                    foreach ($dbRowIterator as $dbRow) {
            
                        $reportRow = [
                            $dbRow['IDLead'],
                            $dbRow['Matricula'],
                            $dbRow['NombreGestor'],
                            $dbRow['FechaHoraReg'],
                            $dbRow['FechaHoraCotizacion'],
                            $dbRow['Gestion'],
                            $dbRow['ResultadoGestion'],
                            $dbRow['Campaña'],
                            $dbRow['Nombres'],
                            $dbRow['Apellidos'],
                            $dbRow['Telefono'],
                            $dbRow['Telefono2'],
                            $dbRow['Origen'],
                            $dbRow['LeadID'],
                            $dbRow['Departamento'],
                            $dbRow['Provincia'],
                            $dbRow['Distrito'],
                            $dbRow['Observaciones'],
                            $dbRow['Segmento'],
                            $dbRow['NLocales'],
                            $dbRow['GGCC'],
                            $dbRow['NDocumento'],
                            $dbRow['Vueltas'],
                            $dbRow['FechaUltimaGestion'],
                            $dbRow['Remarketing'],
                            $dbRow['MatriculaRMKT'],
                            $dbRow['RMKTNombreGestor'],
                            $dbRow['FechaHoraRemarketing'],
                            $dbRow['Gestionable'],
                            $dbRow['SubTipificacion'],
                            $dbRow['ultimoGestor'],
                            $dbRow['ultimoNombreGestor'],
                            $dbRow['Zonas'],
                            $dbRow['Canal'],
                            $dbRow['IdVenta'],
                            $dbRow['TipoVenta'],
                            $dbRow['GestorTV'],
                            $dbRow['EstadoVisita'],
                            $dbRow['FechaCreacionVisita'],
                            $dbRow['FechaVisita'],
                            $dbRow['NProspecto'],
                            $dbRow['Correo'],
                            $dbRow['TipoTransferencia'],
                            $dbRow['Gestionabilidad'],
                            $dbRow['ObsTransferencia'],
                            $dbRow['Contactado'],
                            $dbRow['DevolverLlamada'],
                            $dbRow['FechaUltimaGestionRemarketing'],
                            $dbRow['ResultadoGestionRemarketing'],
                            $dbRow['SubtificacionRemarketing'],
                             $dbRow['visePiso'],
                             $dbRow['rubroNeg'],
                             $dbRow['puertaCalle'],
                             $dbRow['nobleParedes'],
                             $dbRow['nobleTecho'],
                             $dbRow['metros'],
                             $dbRow['motivoInteres'],
                             $dbRow['conSistema'],
                             $dbRow['empresaSistema'],
                             $dbRow['bancarizado'],
                             $dbRow['IdLeadCCREF']
                    ];
            $this->reportWriter->writeRow($reportRow);
        }
    }    
    // 

     //Funciones de Descarga LEADRMKT
         public function fetchDataAndCreateReport_LEADRMKT($outputPath,$fechagrabarfinicial,$fechagrabarffinal){
           $this->reportWriter = $this->createReportWriter($outputPath);
        $this->writeReportHeader_LEADRMKT();

        // Make sure to only select the fields we are interested in
        $query = "SELECT 
                l.IdLeadCC as IDLead, 
                u.Matricula,
                Concat(u.Nombres,' ',u.Apellidos) as NombreGestor,
                l.FechaHoraReg,
                l.FechaHoraCotizacion,
                l.Gestion,
                l.ResultadoGestion,
                l.Campaña,
                l.Nombres,
                l.Apellidos,
                l.Telefono,
                l.Telefono2,
                l.Origen,
                l.LeadID,
                l.Departamento,
                l.Provincia,
                l.Distrito,
                l.Observaciones,
                l.Segmento,
                l.NLocales,
                l.GGCC,
                l.NDocumento,
                l.Vueltas,
                l.FechaUltimaGestion,
                l.Remarketing,
                uRMKT.Matricula as MatriculaRMKT,
                Concat(uRMKT.Nombres,' ',uRMKT.Apellidos) as RMKTNombreGestor,
                l.FechaHoraRemarketing,
                gl.Gestionable,
                gl.SubTipificacion,
                uGest.Matricula as ultimoGestor,
                Concat(uGest.Nombres,' ',uGest.Apellidos) as ultimoNombreGestor,
                d.Zonas,
                CASE
                    WHEN l.Gestion = 'inbound' THEN o.Canal
                ELSE 
	                CASE WHEN l.Origen ='GOOGLE_MARCA' THEN 'SEM MARCA' WHEN l.Origen ='SEO' THEN 'SEM MARCA' ELSE o.Canal END
                END AS Canal,
                v.IdVenta,
                v.TipoVenta,
                l.GestorTV,
                v.EstadoVisita,
                v.FechaHoraREG as FechaCreacionVisita,
                v.FechaVisita,
                v.NProspecto,
                c.Correo,
                l.TipoTransferencia,
                l.Gestionabilidad,
                l.ObsTransferencia
            FROM yifi58ge_dbVENTAS.tblLeadCC l
            INNER JOIN yifi58ge_dbUSUARIO.tblusuario u ON l.IdUsuario = u.IdUsuario
            LEFT JOIN yifi58ge_dbVENTAS.tblGestionLead gl on gl.IdGestionLead=l.UltimaGestion
            LEFT JOIN yifi58ge_dbUSUARIO.tblusuario uGest on gl.IdUsuario = uGest.IdUsuario
            LEFT JOIN yifi58ge_dbVENTAS.tbldistrital d ON d.CodUbigeoReniec=l.CodigoUbigeo
            LEFT JOIN yifi58ge_dbVENTAS.tblorigen o on o.Origen=l.Origen
            LEFT JOIN yifi58ge_dbUSUARIO.tblusuario uRMKT on l.IdUsuarioRemarketing = uRMKT.IdUsuario
            LEFT JOIN (SELECT * FROM yifi58ge_dbVENTAS.tblventa WHERE IdLead IS NOT NULL) v ON l.IdLeadCC = v.IdLead
            LEFT JOIN yifi58ge_dbVENTAS.tblcliente c on c.IdCliente=v.IdCliente
            where l.Remarketing='SI' and l.FechaHoraCotizacion BETWEEN '".$fechagrabarfinicial." 00:00:00' AND '".$fechagrabarffinal." 23:59:59'";

        switch ($this->fetchingMethod) {
            case self::FETCH_ROWS_ONE_BY_ONE:
                $this->fetchRowsOneByOneAndWriteThem_LEADRMKT($query);
                break;
            case self::FETCH_ROWS_IN_BATCH:
                $this->fetchRowsInBatchAndWriteThem($query);
                break;
            case self::FETCH_ROWS_ALL_AT_ONCE:
                $this->fetchAllRowsAtOnceAndWriteThem($query);
                break;
        }

        $this->reportWriter->close();
    }
    private function writeReportHeader_LEADRMKT(){
        // The header will be bold
    $headerRow = [
    'IDLead',
    'Matricula',
    'NombreGestor',
    'FechaHoraReg',
    'FechaHoraCotizacion',
    'Gestion',
    'ResultadoGestion',
    'Campaña',
    'Nombres',
    'Apellidos',
    'Telefono',
    'Telefono2',
    'Origen',
    'LeadID',
    'Departamento',
    'Provincia',
    'Distrito',
    'Observaciones',
    'Segmento',
    'NLocales',
    'GGCC',
    'NDocumento',
    'Vueltas',
    'FechaUltimaGestion',
    'Remarketing',
    'MatriculaRMKT',
    'RMKTNombreGestor',
    'FechaHoraRemarketing',
    'Gestionable',
    'SubTipificacion',
    'ultimoGestor',
    'ultimoNombreGestor',
    'Zonas',
    'Canal',
    'IdVenta',
    'TipoVenta',
    'GestorTV',
    'EstadoVisita',
    'FechaCreacionVisita',
    'FechaVisita',
    'NProspecto',
    'Correo',
    'TipoTransferencia',
    'Gestionabilidad',
    'ObsTransferencia'
        ];
        $this->reportWriter->writeHeaderRow($headerRow);
    }

   private function fetchRowsOneByOneAndWriteThem_LEADRMKT($query){
 $dbRowIterator = new SingleFetchIterator($this->pdo, $query);
            
                    foreach ($dbRowIterator as $dbRow) {
            
                        $reportRow = [
                            $dbRow['IDLead'],
                            $dbRow['Matricula'],
                            $dbRow['NombreGestor'],
                            $dbRow['FechaHoraReg'],
                            $dbRow['FechaHoraCotizacion'],
                            $dbRow['Gestion'],
                            $dbRow['ResultadoGestion'],
                            $dbRow['Campaña'],
                            $dbRow['Nombres'],
                            $dbRow['Apellidos'],
                            $dbRow['Telefono'],
                            $dbRow['Telefono2'],
                            $dbRow['Origen'],
                            $dbRow['LeadID'],
                            $dbRow['Departamento'],
                            $dbRow['Provincia'],
                            $dbRow['Distrito'],
                            $dbRow['Observaciones'],
                            $dbRow['Segmento'],
                            $dbRow['NLocales'],
                            $dbRow['GGCC'],
                            $dbRow['NDocumento'],
                            $dbRow['Vueltas'],
                            $dbRow['FechaUltimaGestion'],
                            $dbRow['Remarketing'],
                            $dbRow['MatriculaRMKT'],
                            $dbRow['RMKTNombreGestor'],
                            $dbRow['FechaHoraRemarketing'],
                            $dbRow['Gestionable'],
                            $dbRow['SubTipificacion'],
                            $dbRow['ultimoGestor'],
                            $dbRow['ultimoNombreGestor'],
                            $dbRow['Zonas'],
                            $dbRow['Canal'],
                            $dbRow['IdVenta'],
                            $dbRow['TipoVenta'],
                            $dbRow['GestorTV'],
                            $dbRow['EstadoVisita'],
                            $dbRow['FechaCreacionVisita'],
                            $dbRow['FechaVisita'],
                            $dbRow['NProspecto'],
                            $dbRow['Correo'],
                            $dbRow['TipoTransferencia'],
                            $dbRow['Gestionabilidad'],
                            $dbRow['ObsTransferencia']
                    ];
            $this->reportWriter->writeRow($reportRow);
        }
    }    
    // 

    private function convertirMicrozona($microzona){
        $nuevaMicrozona = '';
        
        if(strpos($microzona, '-')){
            
            $arrayMicrozona = explode("-", $microzona);
            
            $ubigeo = $arrayMicrozona[0];
            
            $correlativo = '';
            
            $arregloUbigeos = array('150132',
                                '150133',
                                '130101',
                                '130104',
                                '130105',
                                '130111',
                                '130103',
                                '130106',
                                '130107',
                                '130109',
                                '130102',
                                '200101',
                                '200104',
                                '200105',
                                '200107',
                                '200108',
                                '200109',
                                '200110',
                                '200111',
                                '200114',
                                '200115',
                                '040122',
                                '040117',
                                '040112',
                                '040129',
                                '040128',
                                '040104',
                                '040103',
                                '040126',
                                '040101',
                                '040111',
                                '040116',
                                '040107',
                                '040109',
                                '040110',
                                '040123',
                                '040124',
                                '040102',
                                '040120',
                                '040118',
                                '040108',
                                '040125',
                                '040121',
                                '040114',
                                '040113',
                                '040119',
                                '040105',
                                '040106');
                                
            //var_dump($arregloUbigeos);
            
            //exit();
                                
            if (in_array($ubigeo, $arregloUbigeos)) {
                $correlativo = $arrayMicrozona[1];
                
                $correlativo = substr($correlativo, 1, 4);
                
                $nuevaMicrozona = $ubigeo.$correlativo;
                
                
                
            } else{
                $correlativo = $arrayMicrozona[1];
                
                $correlativo = substr($correlativo, 2, 4);
                
                $nuevaMicrozona = $ubigeo.$correlativo;
                
            }
            
            return $nuevaMicrozona;
            
            //return true;    
        } else{
            //return false;
             return $microzona;
        }
    }


    //FUNCIONES DE DESCARGA DE IncidenciasVenCertificadas
    public function fetchDataAndCreateReport_IncidenciasVenCertificadas($outputPath){
        $this->reportWriter = $this->createReportWriter($outputPath);
        $this->writeReportHeader_IncidenciasVenCertificadas();
        // Make sure to only select the fields we are interested in
        $query="SELECT
        	v.IdVenta,
            v.CC,
            v.EstadoVenta,
            v.TipoVenta,
            v.NProspecto,
            v.NContrato,
            v.N590,
            v.HojaCertificacion,
            v.CertificadoValidado,
            v.GestorCertVenta,
            CONCAT(cer.Nombres,' ',cer.Apellidos) as NombGestorCertVenta,
            v.FechaApertura,
            v.IdCertVenta,
            vtacert.FechaCierre,
            vtacert.TipoCierre,
            vtacert.MotivoIncidencia as MotivoIncidenciaCert,
            vtacert.SolucionIncidencia,
            vtacert.IncidenciaTecnica,
            vtacert.FormaPago as FormaPagoCert,
            vtacert.Comentario as ComentarioCert,
            vtacert.kitnoinstalado,
            vtacert.fechacorrecion_kitnoinstalado,
            vtacert.errorcomprobantepago,
            vtacert.fechacorrecion_errorcomprobantepago,
            vtacert.errorsegmento,
            vtacert.fechacorrecion_errorsegmento,
            vtacert.fueranomalcomercial,
            vtacert.fechacorrecion_fueranomalcomercial,
            vtacert.fueranormativatecnica,
            vtacert.fechacorrecion_fueranormativatecnica,
            vtacert.incidenciadatos,
            vtacert.fechacorrecion_incidenciadatos,
            vtacert.inconformidadfuncionamiento,
            vtacert.fechacorrecion_inconformidadfuncionamiento,
            vtacert.inconformidadpagoinicial,
            vtacert.fechacorrecion_inconformidadpagoinicial,
            vtacert.inconformidadpagomensual,
            vtacert.fechacorrecion_inconformidadpagomensual,
            vtacert.inconformidadtiempopermanencia,
            vtacert.fechacorrecion_inconformidadtiempopermanencia,
            vtacert.inconformidadubicaciondispositivos,
            vtacert.fechacorrecion_inconformidadubicaciondispositivos,
            vtacert.noconcefuncionamiento,
            vtacert.fechacorrecion_noconcefuncionamiento,
            vtacert.notienecontrato,
            vtacert.fechacorrecion_notienecontrato,
            vtacert.promesaspendientes,
            vtacert.fechacorrecion_promesaspendientes,
            vtacert.reclamocomercial,
            vtacert.fechacorrecion_reclamocomercial,
            vtacert.solicitabaja,
            vtacert.fechacorrecion_solicitabaja,
            vtacert.tarjetaafiliadanopertenecetitular,
            vtacert.fechacorrecion_tarjetaafiliadanopertenecetitular,
            vtacert.recepmanual,
            vtacert.fechacorrecion_recepmanual,
            vtacert.descargaapp,
            vtacert.fechacorrecion_descargaapp,
            vtacert.capacitado,
            vtacert.fechacorrecion_capacitado,
            vtacert.numerollamadas,
            vtacert.FechaPendiente,
            vtacert.FechaPrimeraGestion,
            vtacert.FechaFinalizadoconIncidencia,
            vtacert.FechaResuelto,
            vtacert.ComentarioComercial,
            vtacert.RespComercial,
            vtacert.nuevohorarioSugerido
        FROM yifi58ge_dbVENTAS.tblventa v
        INNER JOIN yifi58ge_dbVENTAS.tblcertificacionventa vtacert on vtacert.IdCertVenta=v.IdCertVenta
        INNER JOIN yifi58ge_dbUSUARIO.tblusuario cer on cer.Matricula=v.GestorCertVenta     
        ";
        switch ($this->fetchingMethod) {
            case self::FETCH_ROWS_ONE_BY_ONE:
                $this->fetchRowsOneByOneAndWriteThem_IncidenciasVenCertificadas($query);
                break;
            case self::FETCH_ROWS_IN_BATCH:
                $this->fetchRowsInBatchAndWriteThem($query);
                break;
            case self::FETCH_ROWS_ALL_AT_ONCE:
                $this->fetchAllRowsAtOnceAndWriteThem($query);
                break;
        }

        $this->reportWriter->close();
    }
    private function writeReportHeader_IncidenciasVenCertificadas(){
        // The header will be bold
        $headerRow = [
            'IdVenta',
            'CC',
            'EstadoVenta',
            'TipoVenta',
            'NProspecto',
            'NContrato',
            'N590',
            'HojaCertificacion',
            'CertificadoValidado',
            'GestorCertVenta',
            'NombGestorCertVenta',
            'FechaApertura',
            'IdCertVenta',
            'FechaCierre',
            'TipoCierre',
            'MotivoIncidenciaCert',
            'SolucionIncidencia',
            'IncidenciaTecnica',
            'FormaPagoCert',
            'ComentarioCert',
            'kitnoinstalado',
            'fechacorrecion_kitnoinstalado',
            'errorcomprobantepago',
            'fechacorrecion_errorcomprobantepago',
            'errorsegmento',
            'fechacorrecion_errorsegmento',
            'fueranomalcomercial',
            'fechacorrecion_fueranomalcomercial',
            'fueranormativatecnica',
            'fechacorrecion_fueranormativatecnica',
            'incidenciadatos',
            'fechacorrecion_incidenciadatos',
            'inconformidadfuncionamiento',
            'fechacorrecion_inconformidadfuncionamiento',
            'inconformidadpagoinicial',
            'fechacorrecion_inconformidadpagoinicial',
            'inconformidadpagomensual',
            'fechacorrecion_inconformidadpagomensual',
            'inconformidadtiempopermanencia',
            'fechacorrecion_inconformidadtiempopermanencia',
            'inconformidadubicaciondispositivos',
            'fechacorrecion_inconformidadubicaciondispositivos',
            'noconcefuncionamiento',
            'fechacorrecion_noconcefuncionamiento',
            'notienecontrato',
            'fechacorrecion_notienecontrato',
            'promesaspendientes',
            'fechacorrecion_promesaspendientes',
            'reclamocomercial',
            'fechacorrecion_reclamocomercial',
            'solicitabaja',
            'fechacorrecion_solicitabaja',
            'tarjetaafiliadanopertenecetitular',
            'fechacorrecion_tarjetaafiliadanopertenecetitular',
            'recepmanual',
            'fechacorrecion_recepmanual',
            'descargaapp',
            'fechacorrecion_descargaapp',
            'capacitado',
            'fechacorrecion_capacitado',
            'numerollamadas',
            'FechaPendiente',
            'FechaPrimeraGestion',
            'FechaFinalizadoconIncidencia',
            'FechaResuelto',
            'ComentarioComercial',
            'RespComercial',
            'nuevohorarioSugerido'
            ];
        $this->reportWriter->writeHeaderRow($headerRow);
    }
    private function fetchRowsOneByOneAndWriteThem_IncidenciasVenCertificadas($query){
        $dbRowIterator = new SingleFetchIterator($this->pdo, $query);
        foreach ($dbRowIterator as $dbRow) {
        $reportRow = [
            $dbRow['IdVenta'],
            $dbRow['CC'],
            $dbRow['EstadoVenta'],
            $dbRow['TipoVenta'],
            $dbRow['NProspecto'],
            $dbRow['NContrato'],
            $dbRow['N590'],
            $dbRow['HojaCertificacion'],
            $dbRow['CertificadoValidado'],
            $dbRow['GestorCertVenta'],
            $dbRow['NombGestorCertVenta'],
            $dbRow['FechaApertura'],
            $dbRow['IdCertVenta'],
            $dbRow['FechaCierre'],
            $dbRow['TipoCierre'],
            $dbRow['MotivoIncidenciaCert'],
            $dbRow['SolucionIncidencia'],
            $dbRow['IncidenciaTecnica'],
            $dbRow['FormaPagoCert'],
            $dbRow['ComentarioCert'],
            $dbRow['kitnoinstalado'],
            $dbRow['fechacorrecion_kitnoinstalado'],
            $dbRow['errorcomprobantepago'],
            $dbRow['fechacorrecion_errorcomprobantepago'],
            $dbRow['errorsegmento'],
            $dbRow['fechacorrecion_errorsegmento'],
            $dbRow['fueranomalcomercial'],
            $dbRow['fechacorrecion_fueranomalcomercial'],
            $dbRow['fueranormativatecnica'],
            $dbRow['fechacorrecion_fueranormativatecnica'],
            $dbRow['incidenciadatos'],
            $dbRow['fechacorrecion_incidenciadatos'],
            $dbRow['inconformidadfuncionamiento'],
            $dbRow['fechacorrecion_inconformidadfuncionamiento'],
            $dbRow['inconformidadpagoinicial'],
            $dbRow['fechacorrecion_inconformidadpagoinicial'],
            $dbRow['inconformidadpagomensual'],
            $dbRow['fechacorrecion_inconformidadpagomensual'],
            $dbRow['inconformidadtiempopermanencia'],
            $dbRow['fechacorrecion_inconformidadtiempopermanencia'],
            $dbRow['inconformidadubicaciondispositivos'],
            $dbRow['fechacorrecion_inconformidadubicaciondispositivos'],
            $dbRow['noconcefuncionamiento'],
            $dbRow['fechacorrecion_noconcefuncionamiento'],
            $dbRow['notienecontrato'],
            $dbRow['fechacorrecion_notienecontrato'],
            $dbRow['promesaspendientes'],
            $dbRow['fechacorrecion_promesaspendientes'],
            $dbRow['reclamocomercial'],
            $dbRow['fechacorrecion_reclamocomercial'],
            $dbRow['solicitabaja'],
            $dbRow['fechacorrecion_solicitabaja'],
            $dbRow['tarjetaafiliadanopertenecetitular'],
            $dbRow['fechacorrecion_tarjetaafiliadanopertenecetitular'],
            $dbRow['recepmanual'],
            $dbRow['fechacorrecion_recepmanual'],
            $dbRow['descargaapp'],
            $dbRow['fechacorrecion_descargaapp'],
            $dbRow['capacitado'],
            $dbRow['fechacorrecion_capacitado'],
            $dbRow['numerollamadas'],
            $dbRow['FechaPendiente'],
            $dbRow['FechaPrimeraGestion'],
            $dbRow['FechaFinalizadoconIncidencia'],
            $dbRow['FechaResuelto'],
            $dbRow['ComentarioComercial'],
            $dbRow['RespComercial'],
            $dbRow['nuevohorarioSugerido']            
            ];
            $this->reportWriter->writeRow($reportRow);
        }
    }       

  //FIN IncidenciasVenCertificadas



    
}
